function make_getk_text(sectors, countries, rewrite)
% Generate text files that contain parts of the getk function.
% These files are then included in getk.m.
%
% INPUTS:
% - sectors      [integer]   vector of sector indices
% - countries    [cell]      cell array of country names (char arrays)
% - rewrite      [logical]   if true, rewrite the text files even if they already exist

    sectors = num2cell(sectors);

    if nargin<3
        rewrite = false;
    end

    M = numel(sectors);
    N = numel(countries);

    afun = @(x) [ '@(k) ' x(2:end)];

    %
    % Write K_co for beginning of getk
    %

    if ~isfile('getk_text1.m') || rewrite
        fid = fopen('getk_text1.m', 'w');
        for i = 1:N
            fprintf(fid, 'K_%s = @(k) k(%u);\n', countries{i}, i);
        end
        fclose(fid);
    end

    %
    % Write middle part of getk
    %

    if ~isfile('getk_text2.m') || rewrite
        fid = fopen('getk_text2.m', 'w');
        for i = 1:N
            country = countries{i};
            % Write Consumption and Investment anonymous functions (country specific)
            fprintf(fid, 'I_%s = @(k) delta*K_%s(k);\n', country, country);
            fprintf(fid, 'C_%s = @(k) Y_VA_%s_t - I_%s(k);\n', country, country, country);
            % For each sector write Consumption and Investment anonymous functions (country specific)
            for j = 1:M
                fprintf(fid, 'C_%u_%s = @(k) Psi_con_%u_%s*C_%s(k);\n', j, country, j, country, country);
                fprintf(fid, 'I_%u_%s = @(k) Psi_inv_%u_%s*I_%s(k);\n', j, country, j, country, country);
            end
            % Write H_use_{sector}_{country} anonymous functions.
            tmp0 = sprintf('+H_use_%%u_%%%%u_%s(k)', country);
            tmp0 = afun(sprintf(tmp0, sectors{:}));
            for j = 1:M
                A = num2cell(sectors{j}*ones(1,M));
                tmp1 = sprintf(tmp0, A{:});
                tmp1 = sprintf('H_use_%u_%s = %s', sectors{j}, country, tmp1);
                fprintf(fid, '%s;\n', tmp1);
            end
        end
        fclose(fid);
    end

    %
    % Write objective functions of getk
    %

    if ~isfile('getk_text3.m') || rewrite
        fid = fopen('getk_text3.m','w');
        fprintf(fid, 'fun = @(k) [');
        for i = 1:N
            country = countries{i};
            tmp0 = sprintf('K_%s(k)^upsi_K_%s', country, country);
            tmp1 = sprintf('omega_K_1_%s(k)^(1-upsi_K_%s)*K_1_%s(k)^upsi_K_%s', country, country, country, country);
            for j=2:M
                tmp = sprintf('omega_K_%u_%%s(k)^(1-upsi_K_%%s)*K_%u_%%s(k)^upsi_K_%%s', j, j);
                tmp = sprintf(tmp, country, country, country, country);
                tmp1 = sprintf('%s+%s', tmp1, tmp);
            end
            tmp2 = sprintf('%s-(%s)', tmp0, tmp1);
            fprintf(fid, '%s;\n', tmp2);
        end
        for i = 1:N
            country = countries{i};
            for j = 1:M
                for l = 1:M
                    if i*j*l<N*M^2
                        fprintf(fid, 'H_use_%u_%u_%s(k)-H_%u_%u_%s(k);\n', j, l, country, j, l, country);
                    else
                        fprintf(fid, 'H_use_%u_%u_%s(k)-H_%u_%u_%s(k)];\n', j, l, country, j, l, country);
                    end
                end
            end
        end
        fclose(fid);
    end

end
