function [Data_final] = aggregate_countries_SEA(ISO_a, ISO_b, ISO_c, ISO_d, ISO_regions, NACE, Data, Exchange)
%AGGREGATE_COUNTRIES_SEA Aggregate Socio-Economic Accounts data by region
%
%   Data_final = AGGREGATE_COUNTRIES_SEA(ISO_a, ISO_b, ISO_c, ISO_d,
%   ISO_regions, NACE, Data, Exchange) aggregates SEA data from individual
%   countries into regional totals and computes production function parameters.
%
% INPUTS:
%   ISO_a       - Cell array of ISO codes for Region A
%   ISO_b       - Cell array of ISO codes for Region B
%   ISO_c       - Cell array of ISO codes for Region C
%   ISO_d       - Cell array of ISO codes for Region D
%   ISO_regions - String array of region names (e.g., ["Reg_a", "Reg_b", ...])
%   NACE        - String array of NACE sector codes
%   Data        - Structure with SEA data by country from extract_SEA_sectors
%   Exchange    - Table of exchange rates to USD (country x year)
%
% OUTPUTS:
%   Data_final  - Structure containing for each region and sector:
%     .VA, .GO, .II, .LAB, .COMP, .EMP, .K - Aggregated SEA variables
%     .alpha_H       - Value added share (1 - intermediate input share)
%     .alpha_N       - Labor income share in value added
%     .omega_N       - Sector labor share of regional labor supply
%     .omega_K       - Sector capital share of regional capital stock
%     .size          - Sector share of regional value added
%     .Labor_supply  - Total regional labor supply (sum of EMP)
%     .Capital_supply - Total regional capital stock (sum of K)
%     .Value_added   - Total regional value added (sum of VA)
%
% NOTES:
%   - Converts national currencies to USD using exchange rates before aggregation
%   - Excludes 'ROW' countries (no exchange rate available)
%   - Variables are summed across countries; parameters computed from aggregates
%
% REFERENCE:
%   See Appendix A.3 in Hinterlang et al. (2023) for detailed methodology
%
% See also: EXTRACT_SEA_SECTORS, AGGREGATE_COUNTRIES_WIOD

%% Input validation
if ~iscell(ISO_a)
    error('aggregate_countries_SEA:InvalidISOa', ...
        'ISO_a must be a cell array of country codes for Region A');
end

if ~iscell(ISO_b)
    error('aggregate_countries_SEA:InvalidISOb', ...
        'ISO_b must be a cell array of country codes for Region B');
end

if ~iscell(ISO_c)
    error('aggregate_countries_SEA:InvalidISOc', ...
        'ISO_c must be a cell array of country codes for Region C');
end

if ~iscell(ISO_d)
    error('aggregate_countries_SEA:InvalidISOd', ...
        'ISO_d must be a cell array of country codes for Region D');
end

if isempty(ISO_regions)
    error('aggregate_countries_SEA:EmptyRegions', ...
        'ISO_regions cannot be empty');
end

if isempty(NACE)
    error('aggregate_countries_SEA:EmptyNACE', ...
        'NACE sector list cannot be empty');
end

if ~isstruct(Data)
    error('aggregate_countries_SEA:InvalidData', ...
        'Data must be a structure with SEA data by country');
end

if ~istable(Exchange)
    error('aggregate_countries_SEA:InvalidExchange', ...
        'Exchange must be a table of exchange rates (country x year)');
end

vars = {'VA', 'GO', 'II', 'LAB', 'COMP', 'EMP', 'K' };

%% Translate national currencies into USD for variables where necessary

ISO_curr = [ISO_a, ISO_b, ISO_c, ISO_d];

if length(ISO_curr) ==44
ISO_curr = ISO_curr(1:43); % no exchange rate for 'ROW'
end

vars_curr = {'VA', 'GO', 'II', 'LAB', 'COMP',  'K' };

for i = 1:length(ISO_curr)
for j = 1:length(NACE)
     for k =1: length(vars_curr)
     Data.(ISO_curr{i}).(NACE(j)).(vars_curr{k}) = Data.(ISO_curr{i}).(NACE(j)).(vars_curr{k}) .* Exchange{ISO_curr(i),:}';
     end
end
end

% Initialize
Data_final = struct();

for r=1:length(ISO_regions)

    Data_final.(ISO_regions(r)) = struct();

    for j=1:length(NACE)
    Data_final.(ISO_regions(r)).(NACE(j))=struct();

        for k =1: length(vars)
        Data_final.(ISO_regions(r)).(NACE(j)).(vars{k}) = zeros(15,1);
        end
    end

 Data_final.(ISO_regions(r)).Labor_supply = 0;
 Data_final.(ISO_regions(r)).Capital_supply = 0;
 Data_final.(ISO_regions(r)).Value_added = 0;
end


if ismember('ROW', ISO_a)
ISO_a = ISO_a(1:end-1);    % exclude 'ROW' since not included in SEA
end

if ismember('ROW', ISO_b)
ISO_b = ISO_b(1:end-1);    % exclude 'ROW' since not included in SEA
end

if ismember('ROW', ISO_c)
ISO_c = ISO_c(1:end-1);    % exclude 'ROW' since not included in SEA
end

if ismember('ROW', ISO_d)
ISO_d = ISO_d(1:end-1);    % exclude 'ROW' since not included in SEA
end


 % Aggregate countries of each region

 for i = 1:length(ISO_a)
    for j = 1:length(NACE)
     for k =1: length(vars)
     Data_final.Reg_a.(NACE(j)).(vars{k}) = Data_final.Reg_a.(NACE(j)).(vars{k}) + (Data.(ISO_a{i}).(NACE(j)).(vars{k}));
     end
    end
 end

 for i = 1:length(ISO_b)
    for j = 1:length(NACE)
     for k =1: length(vars)
     Data_final.Reg_b.(NACE(j)).(vars{k}) = Data_final.Reg_b.(NACE(j)).(vars{k}) + (Data.(ISO_b{i}).(NACE(j)).(vars{k}));
     end
    end
 end

 for i = 1:length(ISO_c)
    for j = 1:length(NACE)
     for k =1: length(vars)
     Data_final.Reg_c.(NACE(j)).(vars{k}) = Data_final.Reg_c.(NACE(j)).(vars{k}) + (Data.(ISO_c{i}).(NACE(j)).(vars{k}));
     end
    end
 end

 for i = 1:length(ISO_d)
    for j = 1:length(NACE)
     for k =1: length(vars)
     Data_final.Reg_d.(NACE(j)).(vars{k}) = Data_final.Reg_d.(NACE(j)).(vars{k}) + (Data.(ISO_d{i}).(NACE(j)).(vars{k})) ;
     end
    end
 end



for r=1:length(ISO_regions)
for s=1:length(NACE)
% Total amount of labor per region (summed over sectors)
Data_final.(ISO_regions(r)).Labor_supply = Data_final.(ISO_regions(r)).Labor_supply + Data_final.(ISO_regions(r)).((NACE(s))).EMP;

% Total amount of capital per region (summed over sectors)
Data_final.(ISO_regions(r)).Capital_supply = Data_final.(ISO_regions(r)).Capital_supply + Data_final.(ISO_regions(r)).((NACE(s))).K;

% Total VA per region (summed over sectors)
Data_final.(ISO_regions(r)).Value_added = Data_final.(ISO_regions(r)).Value_added + Data_final.(ISO_regions(r)).((NACE(s))).VA;

end


for s=1:length(NACE)
% Parameters production function
Data_final.(ISO_regions(r)).((NACE(s))).alpha_H = 1-Data_final.(ISO_regions(r)).((NACE(s))).II ./ Data_final.(ISO_regions(r)).((NACE(s))).GO;            % alpha_H = 1 - share intermediate inputs
Data_final.(ISO_regions(r)).((NACE(s))).alpha_N_alpha_H = Data_final.(ISO_regions(r)).((NACE(s))).LAB ./ Data_final.(ISO_regions(r)).((NACE(s))).GO;     % alpha_N * alpha_H = labor income share
Data_final.(ISO_regions(r)).((NACE(s))).alpha_N = Data_final.(ISO_regions(r)).((NACE(s))).alpha_N_alpha_H ./ Data_final.(ISO_regions(r)).((NACE(s))).alpha_H;


% Labor and capital shares
Data_final.(ISO_regions(r)).((NACE(s))).omega_N = Data_final.(ISO_regions(r)).((NACE(s))).EMP ./ Data_final.(ISO_regions(r)).Labor_supply;
Data_final.(ISO_regions(r)).((NACE(s))).omega_K = Data_final.(ISO_regions(r)).((NACE(s))).K ./ Data_final.(ISO_regions(r)).Capital_supply;

% Sector size in terms of VA
Data_final.(ISO_regions(r)).((NACE(s))).size = Data_final.(ISO_regions(r)).((NACE(s))).VA ./ Data_final.(ISO_regions(r)).Value_added;

end
end




end


