function [Data_final] = aggregate_countries_WIOD(ISO_all, ISO_a, ISO_b, ISO_c, ISO_d, ISO_regions, NACE, Data)
%AGGREGATE_COUNTRIES_WIOD Aggregate WIOD data from countries to regions
%
%   Data_final = AGGREGATE_COUNTRIES_WIOD(ISO_all, ISO_a, ISO_b, ISO_c,
%   ISO_d, ISO_regions, NACE, Data) aggregates National Accounts and
%   Input-Output data from individual countries into regional totals and
%   computes calibration parameters.
%
% INPUTS:
%   ISO_all     - Cell array of all ISO country codes in the dataset
%   ISO_a       - Cell array of ISO codes for Region A (e.g., home country)
%   ISO_b       - Cell array of ISO codes for Region B
%   ISO_c       - Cell array of ISO codes for Region C
%   ISO_d       - Cell array of ISO codes for Region D (e.g., rest of world)
%   ISO_regions - String array of region names (e.g., ["Reg_a", "Reg_b", ...])
%   NACE        - String array of NACE sector codes
%   Data        - Structure with NA and IO data by country from extract_WIOD_sectors
%
% OUTPUTS:
%   Data_final  - Structure containing for each region:
%     .NA.(region)           - National accounts data by counterpart region
%     .IO.(region)           - Input-output data by counterpart region
%     .NA.Volumes_total_per_sector - Total volumes by sector
%     .NA.Psi_C_I_G          - Consumption/investment share table (Psi_C, Psi_I, etc.)
%     .IO.Psi_H              - Intermediate input share matrix
%     .NA.Biases_C_I.(region) - Home bias for consumption/investment by region
%     .IO.Biases_hhh.(region) - Home bias for intermediate inputs by region
%
% ALGORITHM:
%   1. Aggregate "inner" counterparts: For each country, sum NA/IO from
%      countries in each region to get regional counterparts
%   2. Aggregate "outer" structure: Sum across countries in each region
%      to get region-to-region flows
%   3. Compute shares: Calculate Psi (expenditure shares) and home biases
%
% NOTES:
%   - Validates that shares sum to 1 using assertions
%   - Home bias = regional share / total share (sums to 1 across regions)
%
% REFERENCE:
%   See Appendix A.3 in Hinterlang et al. (2023) for detailed methodology
%
% See also: EXTRACT_WIOD_SECTORS, AGGREGATE_COUNTRIES_SEA

%% Input validation
if ~iscell(ISO_all) || isempty(ISO_all)
    error('aggregate_countries_WIOD:InvalidISOall', ...
        'ISO_all must be a non-empty cell array of country codes');
end

if ~iscell(ISO_a)
    error('aggregate_countries_WIOD:InvalidISOa', ...
        'ISO_a must be a cell array of country codes for Region A');
end

if ~iscell(ISO_b)
    error('aggregate_countries_WIOD:InvalidISOb', ...
        'ISO_b must be a cell array of country codes for Region B');
end

if ~iscell(ISO_c)
    error('aggregate_countries_WIOD:InvalidISOc', ...
        'ISO_c must be a cell array of country codes for Region C');
end

if ~iscell(ISO_d)
    error('aggregate_countries_WIOD:InvalidISOd', ...
        'ISO_d must be a cell array of country codes for Region D');
end

if isempty(ISO_regions)
    error('aggregate_countries_WIOD:EmptyRegions', ...
        'ISO_regions cannot be empty');
end

if isempty(NACE)
    error('aggregate_countries_WIOD:EmptyNACE', ...
        'NACE sector list cannot be empty');
end

if ~isstruct(Data)
    error('aggregate_countries_WIOD:InvalidData', ...
        'Data must be a structure with NA and IO fields by country');
end

%% Initialize final output structure
for r=1:length(ISO_regions)

Data_final.(ISO_regions(r)) = struct();
Data_final.(ISO_regions(r)).NA = struct();
Data_final.(ISO_regions(r)).IO = struct();

    for r2=1:length(ISO_regions)
    Data_final.(ISO_regions(r)).NA.(ISO_regions(r2))= zeros(length(NACE),5);
    Data_final.(ISO_regions(r)).IO.(ISO_regions(r2))= zeros(length(NACE),length(NACE));
    end
end


%% Pre-convert all tables to arrays ONCE (performance optimization)
% This avoids repeated table2array calls in nested loops
Data_numeric = struct();
for i = 1:length(ISO_all)
    Data_numeric.(ISO_all{i}).NA = struct();
    Data_numeric.(ISO_all{i}).IO = struct();
    counterparts = fieldnames(Data.(ISO_all{i}).NA);
    for c = 1:length(counterparts)
        Data_numeric.(ISO_all{i}).NA.(counterparts{c}) = table2array(Data.(ISO_all{i}).NA.(counterparts{c}));
        Data_numeric.(ISO_all{i}).IO.(counterparts{c}) = table2array(Data.(ISO_all{i}).IO.(counterparts{c}));
    end
end

%% Aggregate "inner" counterparts for all countries
% Initialize
for i=1:length(ISO_all)
Data_final.(ISO_all{i}).NA =struct();
Data_final.(ISO_all{i}).IO =struct();
end


for i=1:length(ISO_all)

    for r=1:length(ISO_regions)
    Data_final.(ISO_all{i}).NA.(ISO_regions(r)) = zeros(length(NACE),5);
    Data_final.(ISO_all{i}).IO.(ISO_regions(r)) = zeros(length(NACE),length(NACE));
    end
end


% Aggregate counterparts using pre-converted numeric arrays
for i=1:length(ISO_all)

    for j=1:length(ISO_a) % Sum over Data.Country.NA.Reg_a
    Data_final.(ISO_all{i}).NA.Reg_a  =  Data_final.(ISO_all{i}).NA.Reg_a + Data_numeric.(ISO_all{i}).NA.(ISO_a{j});
    Data_final.(ISO_all{i}).IO.Reg_a  =  Data_final.(ISO_all{i}).IO.Reg_a + Data_numeric.(ISO_all{i}).IO.(ISO_a{j});
    end


    for j=1:length(ISO_b) % Sum over Data.Country.NA.Reg_b
    Data_final.(ISO_all{i}).NA.Reg_b  =  Data_final.(ISO_all{i}).NA.Reg_b + Data_numeric.(ISO_all{i}).NA.(ISO_b{j});
    Data_final.(ISO_all{i}).IO.Reg_b  =  Data_final.(ISO_all{i}).IO.Reg_b  + Data_numeric.(ISO_all{i}).IO.(ISO_b{j});
    end


    for k=1:length(ISO_c) % Sum over Data.Country.NA.ISO_c
    Data_final.(ISO_all{i}).NA.Reg_c =  Data_final.(ISO_all{i}).NA.Reg_c + Data_numeric.(ISO_all{i}).NA.(ISO_c{k});
    Data_final.(ISO_all{i}).IO.Reg_c =  Data_final.(ISO_all{i}).IO.Reg_c + Data_numeric.(ISO_all{i}).IO.(ISO_c{k});
    end


    for l=1:length(ISO_d) % Sum over Data.Country.NA.ISO_d
    Data_final.(ISO_all{i}).NA.Reg_d =  Data_final.(ISO_all{i}).NA.Reg_d +  Data_numeric.(ISO_all{i}).NA.(ISO_d{l});
    Data_final.(ISO_all{i}).IO.Reg_d =  Data_final.(ISO_all{i}).IO.Reg_d +  Data_numeric.(ISO_all{i}).IO.(ISO_d{l});
    end

end



%% Aggregate "outer structure" countries to regions

for j=1:length(ISO_a)
    for r=1:length(ISO_regions)
    Data_final.Reg_a.NA.(ISO_regions(r)) = Data_final.Reg_a.NA.(ISO_regions(r)) + Data_final.(ISO_a{j}).NA.(ISO_regions(r));
    Data_final.Reg_a.IO.(ISO_regions(r)) = Data_final.Reg_a.IO.(ISO_regions(r)) + Data_final.(ISO_a{j}).IO.(ISO_regions(r));
    end
end

for j=1:length(ISO_b)
    for r=1:length(ISO_regions)
    Data_final.Reg_b.NA.(ISO_regions(r)) = Data_final.Reg_b.NA.(ISO_regions(r)) + Data_final.(ISO_b{j}).NA.(ISO_regions(r));
    Data_final.Reg_b.IO.(ISO_regions(r)) = Data_final.Reg_b.IO.(ISO_regions(r)) + Data_final.(ISO_b{j}).IO.(ISO_regions(r));
    end
end

for j=1:length(ISO_c)
    for r=1:length(ISO_regions)
    Data_final.Reg_c.NA.(ISO_regions(r)) = Data_final.Reg_c.NA.(ISO_regions(r)) + Data_final.(ISO_c{j}).NA.(ISO_regions(r));
    Data_final.Reg_c.IO.(ISO_regions(r)) = Data_final.Reg_c.IO.(ISO_regions(r)) + Data_final.(ISO_c{j}).IO.(ISO_regions(r));
    end
end

for j=1:length(ISO_d)
    for r=1:length(ISO_regions)
    Data_final.Reg_d.NA.(ISO_regions(r)) = Data_final.Reg_d.NA.(ISO_regions(r)) + Data_final.(ISO_d{j}).NA.(ISO_regions(r));
    Data_final.Reg_d.IO.(ISO_regions(r)) = Data_final.Reg_d.IO.(ISO_regions(r)) + Data_final.(ISO_d{j}).IO.(ISO_regions(r));
    end
end


% Delete Data_final.Country to have only region aggregates in there
for j=1:length(ISO_all)
    Data_final = rmfield(Data_final, (ISO_all{j}));
end


%% Compute parameters

for r = 1:length(ISO_regions)

    % Initialize accumulators
    Volumes_total_per_sector_NA = zeros(length(NACE), 5);
    Volumes_total_IO = zeros(length(NACE), length(NACE));
    Data_final.(ISO_regions(r)).NA.Biases_C_I = [];

    % STEP 1: Accumulate all volumes across regions
    for r2 = 1:length(ISO_regions)
        Volumes_total_per_sector_NA = Volumes_total_per_sector_NA + Data_final.(ISO_regions(r)).NA.(ISO_regions(r2));
        Volumes_total_IO = Volumes_total_IO + Data_final.(ISO_regions(r)).IO.(ISO_regions(r2));
    end

    % Store accumulated totals
    Data_final.(ISO_regions(r)).NA.Volumes_total_per_sector = Volumes_total_per_sector_NA;
    Data_final.(ISO_regions(r)).IO.Volumes_total = Volumes_total_IO;

    % STEP 2: Compute shares ONCE (after accumulation is complete)
    % Consumption and investment shares
    Volumes_total = sum(Volumes_total_per_sector_NA, 1);
    Psi_C_I_G = Volumes_total_per_sector_NA ./ Volumes_total;
    assert(abs(sum(Psi_C_I_G(:,1)) - 1) < 1e-10, ...
        'Region %s: Consumption shares (Psi_C) must sum to 1', ISO_regions{r});
    assert(abs(sum(Psi_C_I_G(:,4)) - 1) < 1e-10, ...
        'Region %s: Investment shares (Psi_I) must sum to 1', ISO_regions{r});
    Data_final.(ISO_regions(r)).NA.Psi_C_I_G = array2table(Psi_C_I_G, ...
        'VariableNames', {'Psi_C', 'Psi_Con_NP', 'Psi_G', 'Psi_I', 'Psi_Inventories'}, 'RowNames', NACE);

    % Intermediate input shares
    Volumes_total_per_sector_IO = sum(Volumes_total_IO, 1);
    Psi_H = Volumes_total_IO ./ Volumes_total_per_sector_IO;
    assert(all(abs(sum(Psi_H, 1) - 1) < 1e-10), ...
        'Region %s: Intermediate input shares (Psi_H) must sum to 1 for each sector', ISO_regions{r});
    Data_final.(ISO_regions(r)).IO.Psi_H = array2table(Psi_H, ...
        'VariableNames', NACE, 'RowNames', NACE);

    % STEP 3: Compute biases for each region pair
    for r2 = 1:length(ISO_regions)
        % Consumption and investment biases
        Data_final.(ISO_regions(r)).NA.Biases_C_I.(ISO_regions(r2)) = ...
            Data_final.(ISO_regions(r)).NA.(ISO_regions(r2)) ./ Volumes_total_per_sector_NA;
        Data_final.(ISO_regions(r)).NA.Biases_C_I.(ISO_regions(r2)) = array2table(...
            Data_final.(ISO_regions(r)).NA.Biases_C_I.(ISO_regions(r2)), ...
            'VariableNames', {'HB_C', 'HB_NP', 'HB_G', 'HB_I', 'HB_Inventories'}, 'RowNames', NACE);

        % Intermediate input biases
        Data_final.(ISO_regions(r)).IO.Biases_hhh.(ISO_regions(r2)) = ...
            Data_final.(ISO_regions(r)).IO.(ISO_regions(r2)) ./ Volumes_total_IO;
        Data_final.(ISO_regions(r)).IO.Biases_hhh.(ISO_regions(r2)) = array2table(...
            Data_final.(ISO_regions(r)).IO.Biases_hhh.(ISO_regions(r2)), ...
            'VariableNames', NACE, 'RowNames', NACE);
    end
end

end
