%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                       %
%    THE ENVIRONMENTAL MULTI-SECTOR DSGE MODEL EMuSe                    %
%                                                                       %
%    EMuSe has been developed by the                                    %
%    Directorate General Economics of the Deutsche Bundesbank.          %
%                                                                       %
%    Authors:                                                           %
%    Natascha Hinterlang                                                %
%    Anika Martin                                                       %
%    Oke Röhe                                                           %
%    Nikolai Stähler                                                    %
%    Johannes Strobel                                                   %
%                                                                       %
%    Contact: emuse@bundesbank.de                                       %
%                                                                       %
%    The authors are grateful to their colleagues in DG Economics,      %
%    DG Financial Stability and the Research Centre of Deutsche         %
%    Bundesbank, the Working Group on Econometric Modelling (WGEM) and  %
%    the Working Group on Forecasting (WGF) of the European System of   %
%    Central Banks (ESCB) as well as the members of the informal        %
%    network of modelling experts of the G7 Climate Change Mitigation   %
%    Working Group for their helpful discussions and valuable input     %
%    during the development of EMuSe.                                   %
%                                                                       %
%    If you use the EMuSe model, please cite                            %
%    Natascha Hinterlang, Anika Martin, Oke Röhe,                       %
%    Nikolai Stähler and Johannes Strobel (2023),                       %
%    The Environmental Multi-Sector DSGE model EMuSe:                   %
%    A technical documentation,                                         %
%    Deutsche Bundesbank Technical Paper, No. 03/2023.                  %
%                                                                       %
%                                                                       %
% Access to EMuSe is only granted on the basis set out in the           %
% accompanying End User License Agreement (EULA).                       %
% Downloading, installing or using the EMuSe model implies acceptance   %
% of the EULA. If you do not agree to be bound by these terms,          %
% do not download, install or use the related Software and              %
% documentation. As further outlined in the EULA the Software           %
% is provided "as is", without any representation or warranty of any    %
% kind either express or implied.                                       %
%                                                                       %
% The Software and Derived Work from the Software may only be           %
% distributed and communicated together with a copy of the EULA and the %
% aforementioned notice.                                                %
%                                                                       %
%                                                                       %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function Calibration = compute_emission_intensities(Calibration, CO2, ...
    ISO_a, ISO_b, ISO_c, ISO_d, ISO_regions, NACE, NACE_available, all_sectors)
%COMPUTE_EMISSION_INTENSITIES Calibrate emission intensities for EMuSe model
%
%   Calibration = COMPUTE_EMISSION_INTENSITIES(Calibration, CO2, ...
%       ISO_a, ISO_b, ISO_c, ISO_d, ISO_regions, NACE, NACE_available, all_sectors)
%   calibrates CO2 emission intensity parameters using WIOD environmental
%   accounts data. It computes two emission specifications.
%
% INPUTS:
%   Calibration    - Structure with SEA data containing gross output (GO) by sector
%                    Must contain .SEA.(region).(sector).GO field
%   CO2            - Structure with aggregated CO2 emissions data by region
%                    Must contain .(region) fields with emissions tables
%                    (output of extract_and_aggregate_emissions)
%   ISO_a          - Cell array of ISO codes for region A (unused, kept for compatibility)
%   ISO_b          - Cell array of ISO codes for region B (unused, kept for compatibility)
%   ISO_c          - Cell array of ISO codes for region C (unused, kept for compatibility)
%   ISO_d          - Cell array of ISO codes for region D (unused, kept for compatibility)
%   ISO_regions    - String array of region names (e.g., ["Reg_a", "Reg_b"])
%   NACE           - String array of sector codes to use
%   NACE_available - String array of available sector aggregations (unused, kept for compatibility)
%   all_sectors    - Flag: 1 = use all 54 NACE sectors, 0 = use aggregates
%
% OUTPUT:
%   Calibration - Modified structure with added fields:
%     .Emissions.Const_frac.(region) - Constant fraction: Emiss = k * Y
%         Table of emission intensities (CO2/GO) by sector and year
%     .Emissions.Isoelastic.(region) - Isoelastic form: Emiss = Y^(1-k)
%         Table of elasticity parameters by sector and year
%
% DATA NOTES:
%   - CO2 emissions: WIOD Environmental Accounts (2000-2016)
%   - Gross output: WIOD Socio-Economic Accounts (2000-2014)
%   - Output limited to 2000-2014 (overlap period)
%
% EXAMPLE:
%   load('CO2_emissions.mat');  % loads CO2 variable
%   CO2 = extract_and_aggregate_emissions(CO2, ISO_a, ISO_b, ISO_c, ISO_d, ...
%       ISO_regions, all_sectors, NACE, NACE_available);
%   Calibration = compute_emission_intensities(Calibration, CO2, ...
%       ISO_a, ISO_b, ISO_c, ISO_d, ISO_regions, NACE, NACE_available, all_sectors);
%
% See also: EXTRACT_AND_AGGREGATE_EMISSIONS, MAIN_EMUSE_CALIBRATION_TOOLKIT

%% Input validation
if ~isstruct(Calibration)
    error('compute_emission_intensities:InvalidCalibration', ...
        'Calibration must be a structure.');
end

if ~isfield(Calibration, 'SEA')
    error('compute_emission_intensities:MissingSEA', ...
        'Calibration must contain SEA field with gross output data.');
end

if ~isstruct(CO2)
    error('compute_emission_intensities:InvalidCO2', ...
        'CO2 must be a structure.');
end

if isempty(ISO_regions)
    error('compute_emission_intensities:EmptyRegions', ...
        'ISO_regions cannot be empty.');
end

if isempty(NACE)
    error('compute_emission_intensities:EmptyNACE', ...
        'NACE sector codes cannot be empty.');
end

%% Get CO2 years from data
CO2_years = size(CO2.(ISO_regions{1}), 2);

%% Get dimensions
nr_regions = length(ISO_regions);
nr_sectors = length(NACE);

%% Year labels for output tables
YEAR_LABELS = ["y00", "y01", "y02", "y03", "y04", "y05", "y06", "y07", ...
    "y08", "y09", "y10", "y11", "y12", "y13", "y14"];

%% Compute emissions per output
Calibration.Emissions = struct();
Calibration.Emissions.Const_frac = struct();    % Constant fraction of output: Emiss = k * Y
Calibration.Emissions.Isoelastic = struct();    % Isoelastic form: Emiss = Y^{1-k}

for r = 1:nr_regions
    % Emissions data until 2016, but GO from SEA only until 2014
    Calibration.Emissions.Const_frac.(ISO_regions{r}) = zeros(nr_sectors, CO2_years - 2);
    Calibration.Emissions.Isoelastic.(ISO_regions{r}) = zeros(nr_sectors, CO2_years - 2);

    for s = 1:nr_sectors
        if all_sectors == 1
            % For all 54 sectors, need to rename some codes to match CO2 data format
            NACE_help = NACE;
            NACE_help(5) = "C10-C12";    % Must be renamed: CO2 uses hyphens, NACE uses underscores
            NACE_help(6) = "C13-C15";
            NACE_help(26) = "E37-E39";

            CO2_values = table2array(CO2.(ISO_regions{r})(NACE_help(s), 1:end-2));
            GO_values = Calibration.SEA.(ISO_regions{r}).(NACE{s}).GO';

            Calibration.Emissions.Const_frac.(ISO_regions{r})(s,:) = CO2_values ./ GO_values;
            Calibration.Emissions.Isoelastic.(ISO_regions{r})(s,:) = ones(1, CO2_years - 2) - log(CO2_values) ./ log(GO_values);
        else
            CO2_values = table2array(CO2.(ISO_regions{r})(NACE(s), 1:end-2));
            GO_values = Calibration.SEA.(ISO_regions{r}).(NACE{s}).GO';

            Calibration.Emissions.Const_frac.(ISO_regions{r})(s,:) = CO2_values ./ GO_values;
            Calibration.Emissions.Isoelastic.(ISO_regions{r})(s,:) = ones(1, CO2_years - 2) - log(CO2_values) ./ log(GO_values);
        end
    end

    % Convert to tables with proper row and column names
    Calibration.Emissions.Const_frac.(ISO_regions{r}) = array2table(...
        Calibration.Emissions.Const_frac.(ISO_regions{r}), ...
        'RowNames', NACE, 'VariableNames', YEAR_LABELS);
    Calibration.Emissions.Isoelastic.(ISO_regions{r}) = array2table(...
        Calibration.Emissions.Isoelastic.(ISO_regions{r}), ...
        'RowNames', NACE, 'VariableNames', YEAR_LABELS);
end

end
