function Calibration = compute_wiod_means(Calibration, NACE, ISO_regions, year_codes)
%COMPUTE_WIOD_MEANS Compute multi-year averages of WIOD calibration parameters
%
%   Calibration = COMPUTE_WIOD_MEANS(Calibration, NACE, ISO_regions, year_codes)
%   computes mean values of calibration parameters across multiple years using
%   vectorized 3D array operations.
%
% INPUTS:
%   Calibration - Structure with WIOD data by year (e.g., Calibration.WIOD.y05)
%                 Must contain .WIOD.(year_code).(region).NA and .IO fields
%   NACE        - String array of NACE sector codes (e.g., ["A", "B", "C", ...])
%   ISO_regions - String array of region names (e.g., ["Reg_a", "Reg_b"])
%   year_codes  - Cell array of year code strings (e.g., {'y05', 'y06', ...})
%
% OUTPUT:
%   Calibration - Modified structure with added fields:
%     .WIOD.sum.(region)  - Sum across years (for backward compatibility)
%     .WIOD.mean.(region) - Mean across years:
%       .NA.Psi_C_I_G              - Mean consumption/investment shares (table)
%       .IO.Psi_H                  - Mean intermediate input shares (table)
%       .NA.Biases_C_I.(region)    - Mean consumption/investment home biases (table)
%       .IO.Biases_hhh.(region)    - Mean intermediate input home biases (table)
%
% ALGORITHM:
%   1. Stack yearly data into 3D arrays (sectors x variables x years)
%   2. Compute mean along the year dimension using MATLAB's mean()
%   3. Convert results to tables with proper row/column names
%
% EXAMPLE:
%   Calibration = compute_wiod_means(Calibration, NACE, ISO_regions, year_codes);
%
% See also: MAIN_EMUSE_CALIBRATION_TOOLKIT, AGGREGATE_COUNTRIES_WIOD

%% Input validation
if ~isstruct(Calibration)
    error('compute_wiod_means:InvalidCalibration', ...
        'Calibration must be a structure.');
end

if ~isfield(Calibration, 'WIOD')
    error('compute_wiod_means:MissingWIOD', ...
        'Calibration must contain WIOD field.');
end

if isempty(NACE)
    error('compute_wiod_means:EmptyNACE', ...
        'NACE sector codes cannot be empty.');
end

if isempty(ISO_regions)
    error('compute_wiod_means:EmptyRegions', ...
        'ISO_regions cannot be empty.');
end

if isempty(year_codes)
    error('compute_wiod_means:EmptyYearCodes', ...
        'year_codes cannot be empty.');
end

%% Get dimensions
n_sectors = length(NACE);
n_regions = length(ISO_regions);
n_years = length(year_codes);

%% Initialize output structures
Calibration.WIOD.sum = struct();
Calibration.WIOD.mean = struct();

for r = 1:n_regions
    Calibration.WIOD.sum.(ISO_regions{r}).NA = struct();
    Calibration.WIOD.sum.(ISO_regions{r}).NA.Biases_C_I = struct();
    Calibration.WIOD.sum.(ISO_regions{r}).IO = struct();
    Calibration.WIOD.sum.(ISO_regions{r}).IO.Biases_hhh = struct();

    Calibration.WIOD.mean.(ISO_regions{r}).NA = struct();
    Calibration.WIOD.mean.(ISO_regions{r}).IO = struct();
    Calibration.WIOD.mean.(ISO_regions{r}).NA.Biases_C_I = struct();
    Calibration.WIOD.mean.(ISO_regions{r}).IO.Biases_hhh = struct();
end

%% Compute means using 3D array stacking
for r = 1:n_regions
    % Pre-allocate 3D arrays for stacking yearly data
    Psi_C_I_G_stack = zeros(n_sectors, 5, n_years);
    Psi_H_stack = zeros(n_sectors, n_sectors, n_years);

    Biases_C_I_stack = struct();
    Biases_hhh_stack = struct();
    for r2 = 1:n_regions
        Biases_C_I_stack.(ISO_regions{r2}) = zeros(n_sectors, 5, n_years);
        Biases_hhh_stack.(ISO_regions{r2}) = zeros(n_sectors, n_sectors, n_years);
    end

    % Stack data from all years into 3D arrays
    for i = 1:n_years
        Psi_C_I_G_stack(:,:,i) = table2array(Calibration.WIOD.(year_codes{i}).(ISO_regions{r}).NA.Psi_C_I_G);
        Psi_H_stack(:,:,i) = table2array(Calibration.WIOD.(year_codes{i}).(ISO_regions{r}).IO.Psi_H);

        for r2 = 1:n_regions
            Biases_C_I_stack.(ISO_regions{r2})(:,:,i) = table2array(Calibration.WIOD.(year_codes{i}).(ISO_regions{r}).NA.Biases_C_I.(ISO_regions{r2}));
            Biases_hhh_stack.(ISO_regions{r2})(:,:,i) = table2array(Calibration.WIOD.(year_codes{i}).(ISO_regions{r}).IO.Biases_hhh.(ISO_regions{r2}));
        end
    end

    % Compute sums (for backward compatibility) and means using vectorized operations
    Calibration.WIOD.sum.(ISO_regions{r}).NA.Psi_C_I_G = sum(Psi_C_I_G_stack, 3);
    Calibration.WIOD.sum.(ISO_regions{r}).IO.Psi_H = sum(Psi_H_stack, 3);

    Calibration.WIOD.mean.(ISO_regions{r}).NA.Psi_C_I_G = array2table(mean(Psi_C_I_G_stack, 3), ...
        'VariableNames', {'Psi_C', 'Psi_Con_NP', 'Psi_G', 'Psi_I', 'Psi_Inventories'}, 'RowNames', NACE);
    Calibration.WIOD.mean.(ISO_regions{r}).IO.Psi_H = array2table(mean(Psi_H_stack, 3), ...
        'VariableNames', NACE, 'RowNames', NACE);

    for r2 = 1:n_regions
        Calibration.WIOD.sum.(ISO_regions{r}).NA.Biases_C_I.(ISO_regions{r2}) = sum(Biases_C_I_stack.(ISO_regions{r2}), 3);
        Calibration.WIOD.sum.(ISO_regions{r}).IO.Biases_hhh.(ISO_regions{r2}) = sum(Biases_hhh_stack.(ISO_regions{r2}), 3);

        Calibration.WIOD.mean.(ISO_regions{r}).NA.Biases_C_I.(ISO_regions{r2}) = array2table(mean(Biases_C_I_stack.(ISO_regions{r2}), 3), ...
            'VariableNames', {'HB_C', 'HB_NP', 'HB_G', 'HB_I', 'HB_Inventories'}, 'RowNames', NACE);
        Calibration.WIOD.mean.(ISO_regions{r}).IO.Biases_hhh.(ISO_regions{r2}) = array2table(mean(Biases_hhh_stack.(ISO_regions{r2}), 3), ...
            'VariableNames', NACE, 'RowNames', NACE);
    end
end

end
