function  [A01_Data, A02_Data, A03_Data,...
      B_Data, C10_12_Data, C13_15_Data,...
      C16_Data, C17_Data, C18_Data,...
      C19_Data, C20_Data, C21_Data,...
      C22_Data, C23_Data, C24_Data,...
      C25_Data, C26_Data, C27_Data,...
      C28_Data, C29_Data, C30_Data,...
      C31_32_Data, C33_Data, D35_Data, E36_Data,...
      E37_39_Data, F_Data, G45_Data,...
      G46_Data, G47_Data, H49_Data, H50_Data,...
      H51_Data, H52_Data, H53_Data,...
      I_Data, J58_Data, J59_60_Data,...
      J61_Data, J62_63_Data, K64_Data,...
      K65_Data, K66_Data, L68_Data,...
      M69_70_Data, M71_Data, M72_Data, ...
      M73_Data, M74_75_Data, N_Data,...
      O84_Data, P85_Data, Q_Data,...
      R_S_Data] = extract_SEA_all_sectors(WIOD_SEA_dat,ISO)
%EXTRACT_SEA_ALL_SECTORS Extract Socio-Economic Accounts data for all 54 NACE sectors
%
%   [A01_Data, A02_Data, ...] = EXTRACT_SEA_ALL_SECTORS(WIOD_SEA_dat, ISO)
%   extracts SEA data for a country at the full 54-sector NACE detail level.
%
% INPUTS:
%   WIOD_SEA_dat - Table containing WIOD Socio-Economic Accounts data
%   ISO          - String with ISO country code to extract (e.g., 'DEU')
%
% OUTPUTS:
%   54 structures (one per NACE sector), each containing:
%     .VA, .GO, .II, .LAB, .COMP, .EMP, .H_EMPE, .EMPE, .K
%
% NOTES:
%   - Returns data at the most detailed sector level (no aggregation)
%   - Use extract_SEA_sectors for aggregated sector analysis
%
% See also: EXTRACT_SEA_SECTORS, AGGREGATE_COUNTRIES_SEA
%
%
% WIOD SEA available data:
%
% Values	Description
% GO        Gross output by industry at current basic prices (in millions of national currency)
% II        Intermediate inputs at current purchasers' prices (in millions of national currency)
% VA        Gross value added at current basic prices (in millions of national currency)
% EMP       Number of persons engaged (thousands)
% EMPE      Number of employees (thousands)
% H_EMPE	Total hours worked by employees (millions)
% COMP      Compensation of employees (in millions of national currency)
% LAB       Labour compensation (in millions of national currency)
% CAP       Capital compensation (in millions of national currency)
% K         Nominal capital stock (in millions of national currency)
%
% Prices
% GO_PI     Price levels gross output, 2010=100
% II_PI     Price levels of intermediate inputs, 2010=100
% VA_PI     Price levels of gross value added, 2010=100
%
% Volumes
% GO_QI     Gross output, volume indices, 2010=100
% II_QI     Intermediate inputs, volume indices, 2010=100
% VA_QI     Gross value added, volume indices, 2010=100

%% Input validation
if ~istable(WIOD_SEA_dat) || isempty(WIOD_SEA_dat)
    error('extract_SEA_all_sectors:InvalidSEA', ...
        'WIOD_SEA_dat must be a non-empty table');
end

if ~(ischar(ISO) || isstring(ISO) || iscell(ISO))
    error('extract_SEA_all_sectors:InvalidISO', ...
        'ISO must be a string, char, or cell with a country code (e.g., ''DEU'')');
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Extract country
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

rows = (WIOD_SEA_dat.country==ISO);

SEA_country=WIOD_SEA_dat(rows,:);


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Extract nominal data
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

vars = {'VA', 'GO', 'II', 'LAB', 'EMP', 'K' };
nr_vars = length(vars);

% Define all sector codes (must match function output order)
sector_codes = {'A01', 'A02', 'A03', 'B', 'C10-C12', 'C13-C15', 'C16', 'C17', 'C18', ...
                'C19', 'C20', 'C21', 'C22', 'C23', 'C24', 'C25', 'C26', 'C27', ...
                'C28', 'C29', 'C30', 'C31_C32', 'C33', 'D35', 'E36', 'E37-E39', ...
                'F', 'G45', 'G46', 'G47', 'H49', 'H50', 'H51', 'H52', 'H53', 'I', ...
                'J58', 'J59_J60', 'J61', 'J62_J63', 'K64', 'K65', 'K66', 'L68', ...
                'M69_M70', 'M71', 'M72', 'M73', 'M74_M75', 'N', 'O84', 'P85', 'Q', 'R_S'};
num_sectors = length(sector_codes);

% Pre-compute sector masks ONCE (avoids repeated comparisons in nested loop)
sector_masks = cell(1, num_sectors);
for s = 1:num_sectors
    sector_masks{s} = (SEA_country.code == sector_codes{s});
end

% Get numeric data columns ONCE (exclude metadata columns)
numeric_cols = ~ismember(SEA_country.Properties.VariableNames, {'country', 'variable', 'description', 'code'});
SEA_numeric = SEA_country{:, numeric_cols};

% Initialize output structures using cell array for efficient assignment
sector_outputs = cell(1, num_sectors);
for s = 1:num_sectors
    sector_outputs{s} = struct();
end

% Process all variables and sectors efficiently
for i = 1:nr_vars
    var_mask = (SEA_country.variable == vars(i));

    for s = 1:num_sectors
        % Combine variable mask with pre-computed sector mask
        combined_mask = var_mask & sector_masks{s};
        % Extract and transpose numeric data directly (no removevars/table2array calls)
        sector_outputs{s}.(vars{i}) = SEA_numeric(combined_mask, :)';
    end
end

% Assign to output variables (must match function signature order)
A01_Data = sector_outputs{1}; A02_Data = sector_outputs{2}; A03_Data = sector_outputs{3};
B_Data = sector_outputs{4}; C10_12_Data = sector_outputs{5}; C13_15_Data = sector_outputs{6};
C16_Data = sector_outputs{7}; C17_Data = sector_outputs{8}; C18_Data = sector_outputs{9};
C19_Data = sector_outputs{10}; C20_Data = sector_outputs{11}; C21_Data = sector_outputs{12};
C22_Data = sector_outputs{13}; C23_Data = sector_outputs{14}; C24_Data = sector_outputs{15};
C25_Data = sector_outputs{16}; C26_Data = sector_outputs{17}; C27_Data = sector_outputs{18};
C28_Data = sector_outputs{19}; C29_Data = sector_outputs{20}; C30_Data = sector_outputs{21};
C31_32_Data = sector_outputs{22}; C33_Data = sector_outputs{23}; D35_Data = sector_outputs{24};
E36_Data = sector_outputs{25}; E37_39_Data = sector_outputs{26}; F_Data = sector_outputs{27};
G45_Data = sector_outputs{28}; G46_Data = sector_outputs{29}; G47_Data = sector_outputs{30};
H49_Data = sector_outputs{31}; H50_Data = sector_outputs{32}; H51_Data = sector_outputs{33};
H52_Data = sector_outputs{34}; H53_Data = sector_outputs{35}; I_Data = sector_outputs{36};
J58_Data = sector_outputs{37}; J59_60_Data = sector_outputs{38}; J61_Data = sector_outputs{39};
J62_63_Data = sector_outputs{40}; K64_Data = sector_outputs{41}; K65_Data = sector_outputs{42};
K66_Data = sector_outputs{43}; L68_Data = sector_outputs{44}; M69_70_Data = sector_outputs{45};
M71_Data = sector_outputs{46}; M72_Data = sector_outputs{47}; M73_Data = sector_outputs{48};
M74_75_Data = sector_outputs{49}; N_Data = sector_outputs{50}; O84_Data = sector_outputs{51};
P85_Data = sector_outputs{52}; Q_Data = sector_outputs{53}; R_S_Data = sector_outputs{54};

end












