function [A_tot, B_tot, C_tot, MF_wo_C19_tot, D_tot, E_tot, B_C_D_E_tot, B_D_E_tot, B_C19_D_tot, B_C19_tot, F_tot, E_F_tot, G_H_I_tot, H_tot, J_tot, K_tot, L_tot, M_N_tot, O_P_Q_tot, R_S_tot, Serv_tot, E_F_Serv_wo_H_tot] = extract_SEA_sectors(WIOD_SEA_dat,ISO)
%EXTRACT_SEA_SECTORS Extract and aggregate Socio-Economic Accounts data by sector
%
%   [A_tot, B_tot, ...] = EXTRACT_SEA_SECTORS(WIOD_SEA_dat, ISO) extracts
%   SEA data for a country and aggregates it into predefined sector groups.
%
% INPUTS:
%   WIOD_SEA_dat - Table containing WIOD Socio-Economic Accounts data
%   ISO          - String with ISO country code to extract (e.g., 'DEU')
%
% OUTPUTS:
%   22 structures (A_tot, B_tot, ..., E_F_Serv_wo_H_tot), each containing:
%     .VA    - Gross value added (millions of national currency)
%     .GO    - Gross output (millions of national currency)
%     .II    - Intermediate inputs (millions of national currency)
%     .LAB   - Labour compensation (millions of national currency)
%     .COMP  - Compensation of employees (millions of national currency)
%     .EMP   - Number of persons engaged (thousands)
%     .H_EMPE - Total hours worked by employees (millions)
%     .EMPE  - Number of employees (thousands)
%     .K     - Nominal capital stock (millions of national currency)
%
% SECTOR AGGREGATIONS:
%   A=Agriculture, B=Mining, C=Manufacturing, D=Electricity, E=Water/Waste,
%   F=Construction, G_H_I=Trade/Transport/Accommodation, J=ICT, K=Finance,
%   L=Real Estate, M_N=Professional Services, O_P_Q=Public Admin/Education/Health,
%   R_S=Arts/Other Services, Serv=All Services, etc.
%
% NOTES:
%   - Uses get_sector_mappings() for sector aggregation definitions
%   - Data is summed across constituent NACE sectors for each aggregate
%
% See also: EXTRACT_SEA_ALL_SECTORS, AGGREGATE_COUNTRIES_SEA, GET_SECTOR_MAPPINGS

%% Input validation
if ~istable(WIOD_SEA_dat) || isempty(WIOD_SEA_dat)
    error('extract_SEA_sectors:InvalidSEA', ...
        'WIOD_SEA_dat must be a non-empty table');
end

if ~(ischar(ISO) || isstring(ISO) || iscell(ISO))
    error('extract_SEA_sectors:InvalidISO', ...
        'ISO must be a string, char, or cell with a country code (e.g., ''DEU'')');
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Extract country
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

rows = (WIOD_SEA_dat.country==ISO);

SEA_country=WIOD_SEA_dat(rows,:);


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Extract nominal data
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

vars = {'VA', 'GO', 'II', 'LAB', 'COMP', 'EMP', 'H_EMPE','EMPE', 'K' };
nr_vars = length(vars);
Data=struct();

% Get sector mappings from shared utility function
[sector_map, NACE_55] = get_sector_mappings();

% Define sector names and output variable names (must match function outputs)
sector_names = {'A', 'B', 'C', 'MF_wo_C19', 'D', 'E', 'B_C_D_E', 'B_D_E', 'B_C19_D', 'B_C19', ...
                'F', 'E_F', 'G_H_I', 'H', 'J', 'K', 'L', 'M_N', 'O_P_Q', 'R_S', 'Serv', 'E_F_Serv_wo_H'};

% Initialize output structures
A_tot = struct(); B_tot = struct(); C_tot = struct(); MF_wo_C19_tot = struct();
D_tot = struct(); E_tot = struct(); B_C_D_E_tot = struct(); B_D_E_tot = struct();
B_C19_D_tot = struct(); B_C19_tot = struct(); F_tot = struct(); E_F_tot = struct();
G_H_I_tot = struct(); H_tot = struct(); J_tot = struct(); K_tot = struct();
L_tot = struct(); M_N_tot = struct(); O_P_Q_tot = struct(); R_S_tot = struct();
Serv_tot = struct(); E_F_Serv_wo_H_tot = struct();

% Map sector names to output variables
sector_outputs = {A_tot, B_tot, C_tot, MF_wo_C19_tot, D_tot, E_tot, B_C_D_E_tot, B_D_E_tot, ...
                  B_C19_D_tot, B_C19_tot, F_tot, E_F_tot, G_H_I_tot, H_tot, J_tot, K_tot, ...
                  L_tot, M_N_tot, O_P_Q_tot, R_S_tot, Serv_tot, E_F_Serv_wo_H_tot};

% Pre-compute sector masks ONCE (avoids repeated ismember calls in nested loop)
sector_masks = cell(1, length(sector_names));
for s = 1:length(sector_names)
    sector_indices = sector_map.(sector_names{s});
    sector_codes = NACE_55(sector_indices);
    sector_masks{s} = ismember(SEA_country.code, sector_codes);
end

% Get numeric data columns (exclude metadata columns)
numeric_cols = ~ismember(SEA_country.Properties.VariableNames, {'country', 'variable', 'description', 'code'});
SEA_numeric = SEA_country{:, numeric_cols};

for i = 1:nr_vars
    var_mask = (SEA_country.variable==vars(i));
    Data.(vars{i}) = SEA_country(var_mask,:);

    % Process each sector aggregation using pre-computed masks
    for s = 1:length(sector_names)
        % Combine variable mask with pre-computed sector mask
        combined_mask = var_mask & sector_masks{s};

        % Sum numeric data directly (avoids repeated removevars calls)
        sector_sum = sum(SEA_numeric(combined_mask, :), 1);
        sector_outputs{s}.(vars{i}) = sector_sum';
    end
end

% Assign back to output variables
A_tot = sector_outputs{1}; B_tot = sector_outputs{2}; C_tot = sector_outputs{3};
MF_wo_C19_tot = sector_outputs{4}; D_tot = sector_outputs{5}; E_tot = sector_outputs{6};
B_C_D_E_tot = sector_outputs{7}; B_D_E_tot = sector_outputs{8}; B_C19_D_tot = sector_outputs{9};
B_C19_tot = sector_outputs{10}; F_tot = sector_outputs{11}; E_F_tot = sector_outputs{12};
G_H_I_tot = sector_outputs{13}; H_tot = sector_outputs{14}; J_tot = sector_outputs{15};
K_tot = sector_outputs{16}; L_tot = sector_outputs{17}; M_N_tot = sector_outputs{18};
O_P_Q_tot = sector_outputs{19}; R_S_tot = sector_outputs{20}; Serv_tot = sector_outputs{21};
E_F_Serv_wo_H_tot = sector_outputs{22};

end
