function [NA, IO] = extract_WIOD_all_sectors(WIOD_Data, ISO, ISO2, NACE)
%EXTRACT_WIOD_ALL_SECTORS Extract WIOD data for all 54 NACE sectors
%
%   [NA, IO] = EXTRACT_WIOD_ALL_SECTORS(WIOD_Data, ISO, ISO2, NACE) extracts
%   National Accounts and Input-Output data from the World Input-Output
%   Database for all 54 NACE sectors without aggregation.
%
% INPUTS:
%   WIOD_Data - Cell array containing WIOT table (industry-by-industry,
%               current prices in millions of $)
%   ISO       - String or cell array of ISO country code(s) to extract
%   ISO2      - Cell array of all country codes in the dataset
%   NACE      - String array of all 54 NACE sector codes
%
% OUTPUTS:
%   NA        - Structure with National Accounts data by country
%               Fields per country: Con_HH, Con_NP, CON_Gov, GFCF, Inv
%   IO        - Structure with Input-Output tables by country
%               Tables of intermediate inputs (54x54 sectors)
%
% NOTES:
%   - Sectors T and U are excluded (not reported in all countries)
%   - Use extract_WIOD_sectors for aggregated sector analysis
%
% See also: EXTRACT_WIOD_SECTORS, AGGREGATE_COUNTRIES_WIOD

%% Input validation
if ~iscell(WIOD_Data) || isempty(WIOD_Data)
    error('extract_WIOD_all_sectors:InvalidWIOD', ...
        'WIOD_Data must be a non-empty cell array');
end

if ~(ischar(ISO) || iscell(ISO) || isstring(ISO))
    error('extract_WIOD_all_sectors:InvalidISO', ...
        'ISO must be a string, char, or cell array of country codes');
end

if ~iscell(ISO2) || isempty(ISO2)
    error('extract_WIOD_all_sectors:InvalidISO2', ...
        'ISO2 must be a non-empty cell array of all country codes');
end

if isempty(NACE)
    error('extract_WIOD_all_sectors:EmptyNACE', ...
        'NACE sector list cannot be empty');
end

%% Extract input-output part and national accounts part separately
% Remove 'T' and 'U' sector since not available in all countries
rows_T = ismember( cellstr(WIOD_Data(:,1)), 'T');
columns_T = ismember( cellstr(WIOD_Data(3,:)), 'T');
WIOD_Data(rows_T,:) = [];
WIOD_Data(:,columns_T) = [];

rows_U = ismember( cellstr(WIOD_Data(:,1)), 'U');
columns_U = ismember( cellstr(WIOD_Data(3,:)), 'U');
WIOD_Data(rows_U,:) = [];
WIOD_Data(:,columns_U) = [];

columns = ismember(cellstr(WIOD_Data(5,:)), ISO);
WIOD_country = cell(length(WIOD_Data(:,1)), 4+sum(columns));
WIOD_country(:,1:4) = WIOD_Data(:,1:4);
WIOD_country(:,5:end) = WIOD_Data(:,columns);

NA_vars = {'CONS_h', 'CONS_np', 'CONS_g', 'GFCF', 'INVEN'};
columns_NA = ismember(cellstr(WIOD_country(3,:)), NA_vars);
columns_IO = logical(ones(1, length(columns_NA))-double(columns_NA));

% Input-Output part
WIOD_country_IO = cell(length(WIOD_country(:,1)), sum(columns_IO));
WIOD_country_IO(:, :) = WIOD_country(:, columns_IO);

% National accounts part
WIOD_country_NA = cell(length(WIOD_country(:,1)), 4+sum(columns_NA));
WIOD_country_NA(:,1:4) = WIOD_country(:,1:4);
WIOD_country_NA(:,5:end) = WIOD_country(:,columns_NA);

NACE_help = NACE;
NACE_help(5) = "C10-C12";               % Must be renamed, otherwise does not recognize "C10_C12" etc.
NACE_help(6) = "C13-C15";
NACE_help(26) = "E37-E39";

% Cache cellstr conversions BEFORE loops (performance optimization)
NA_countries_cellstr_full = cellstr(WIOD_country_NA(:,3)); % Country codes (all rows)
NA_sectors_cellstr_full = cellstr(WIOD_country_NA(:,1));   % Sector codes (all rows)

% Identify data rows (rows with valid NACE sector codes, excluding header rows)
data_rows_mask = ismember(NA_sectors_cellstr_full, NACE_help);

% Extract only data rows for cellstr arrays and numeric conversion
NA_countries_cellstr = NA_countries_cellstr_full(data_rows_mask);
NA_sectors_cellstr = NA_sectors_cellstr_full(data_rows_mask);

% Pre-convert cell data to numeric matrices ONCE (only data rows)
NA_data_numeric = cell2mat(WIOD_country_NA(data_rows_mask, 5:end));
IO_data_numeric = cell2mat(WIOD_country_IO(data_rows_mask, 5:end));

% Pre-compute sector masks (avoids repeated ismember calls in inner loop)
sector_masks = cell(1, length(NACE));
for j = 1:length(NACE)
    sector_masks{j} = ismember(NA_sectors_cellstr, NACE_help(j));
end

    for i=1:length(ISO2)
     % Pre-compute country mask once per country
     country_mask = ismember(NA_countries_cellstr, ISO2(i));

     for j=1:length(NACE)
     % Logicals for country-sector combination
     row_mask = country_mask & sector_masks{j};
     rows.(ISO2{i}).(NACE(j)) = row_mask;

     % Aggregate NA using pre-converted numeric matrix
     NA.(ISO2{i})(j,:) = sum(NA_data_numeric(row_mask, :), 1);

     % Aggregate IO using pre-converted numeric matrix
     IO.(ISO2{i})(j,:) = sum(IO_data_numeric(row_mask, :), 1);

     end
     end


    for i=1:length(ISO2)
    NA.(ISO2{i}) = array2table(NA.(ISO2{i}), 'VariableNames',{'Con_HH', 'Con_NP', 'CON_Gov', 'GFCF', 'Inv'}, 'RowNames', NACE);
    IO.(ISO2{i}) = array2table(IO.(ISO2{i}), 'VariableNames', NACE, 'RowNames', NACE);
    end


end








