function [NA, IO] = extract_WIOD_sectors(WIOD_Data, ISO, ISO2, NACE)
%EXTRACT_WIOD_SECTORS Extract WIOD data for specified sectors and countries
%
%   [NA, IO] = EXTRACT_WIOD_SECTORS(WIOD_Data, ISO, ISO2, NACE) extracts
%   National Accounts and Input-Output data from the World Input-Output
%   Database for specified countries and aggregated economic sectors.
%
% INPUTS:
%   WIOD_Data - Cell array containing WIOT table with structure:
%               Rows: Sectors by country, Cols: Sectors by country + final demand
%               Data is industry-by-industry in current prices (millions of $)
%   ISO       - String or cell array of ISO country code(s) to extract
%               Example: 'DEU' or {'DEU', 'FRA'}
%   ISO2      - Cell array of all country codes in the dataset
%               Used for cross-country IO relationships
%   NACE      - String array of aggregated sector codes to extract
%               Example: ["A", "B", "C", "D", "E", "F"]
%               Uses sector mappings from get_sector_mappings()
%
% OUTPUTS:
%   NA        - Structure with National Accounts data by country
%               Each field (country code) contains a table with columns:
%               Con_HH (household consumption), Con_NP (NPISH consumption),
%               CON_Gov (government consumption), GFCF (investment), Inv (inventories)
%   IO        - Structure with Input-Output tables by country
%               Each field contains a table of intermediate inputs (rows: from, cols: to)
%
% NOTES:
%   - Sector U is excluded as it's not reported in all countries
%   - Uses get_sector_mappings() for sector aggregation definitions
%
% EXAMPLE:
%   load('WIOT2005_Nov16_ROW.mat');
%   sectors = ["A", "B", "C", "D", "E", "F"];
%   [na, io] = extract_WIOD_sectors(WIOD, 'DEU', all_countries, sectors);
%
% REFERENCE:
%   See Appendix A.3 in Hinterlang et al. (2023) for detailed methodology
%
% See also: EXTRACT_WIOD_ALL_SECTORS, AGGREGATE_COUNTRIES_WIOD, GET_SECTOR_MAPPINGS

%% Input validation
if ~iscell(WIOD_Data) || isempty(WIOD_Data)
    error('extract_WIOD_sectors:InvalidWIOD', ...
        'WIOD_Data must be a non-empty cell array');
end

if ~(ischar(ISO) || iscell(ISO) || isstring(ISO))
    error('extract_WIOD_sectors:InvalidISO', ...
        'ISO must be a string, char, or cell array of country codes');
end

if ~iscell(ISO2) || isempty(ISO2)
    error('extract_WIOD_sectors:InvalidISO2', ...
        'ISO2 must be a non-empty cell array of all country codes');
end

if isempty(NACE)
    error('extract_WIOD_sectors:EmptyNACE', ...
        'NACE sector list cannot be empty');
end

%% Extract input-output part and national accounts part separately
% Remove U sector since not available in all countries
rows_U = ismember( cellstr(WIOD_Data(:,1)), 'U');
columns_U = ismember( cellstr(WIOD_Data(3,:)), 'U');
WIOD_Data(rows_U,:) = [];
WIOD_Data(:,columns_U) = [];

columns = ismember(cellstr(WIOD_Data(5,:)), ISO);
WIOD_country = cell(length(WIOD_Data(:,1)), 4+sum(columns));
WIOD_country(:,1:4) = WIOD_Data(:,1:4);
WIOD_country(:,5:end) = WIOD_Data(:,columns);

NA_vars = {'CONS_h', 'CONS_np', 'CONS_g', 'GFCF', 'INVEN'};
columns_NA = ismember(cellstr(WIOD_country(3,:)), NA_vars);
columns_IO = logical(ones(1, length(columns_NA))-double(columns_NA));

% Input-Output part
WIOD_country_IO = cell(length(WIOD_country(:,1)), sum(columns_IO));
WIOD_country_IO(:, :) = WIOD_country(:, columns_IO);

% National accounts part
WIOD_country_NA = cell(length(WIOD_country(:,1)), 4+sum(columns_NA));
WIOD_country_NA(:,1:4) = WIOD_country(:,1:4);
WIOD_country_NA(:,5:end) = WIOD_country(:,columns_NA);

%% Aggregate sectors (NACE) for each country (ISO2)

% Get sector mappings from shared utility function
[sector_map, NACE_55] = get_sector_mappings();

% Define the order of sector aggregations
sector_names = {'A', 'B', 'C', 'MF_wo_C19', 'D', 'E', 'B_C_D_E', 'B_D_E', 'B_C19_D', 'B_C19', ...
                'F', 'E_F', 'G_H_I', 'H', 'J', 'K', 'L', 'M_N', 'O_P_Q', 'R_S', 'Serv', 'E_F_Serv_wo_H'};

% Cache cellstr conversions BEFORE loops (performance optimization)
NA_sectors_cellstr_full = cellstr(WIOD_country_NA(:,1));   % Sector codes in NA data (all rows)
NA_countries_cellstr_full = cellstr(WIOD_country_NA(:,3)); % Country codes in NA data (all rows)
IO_sectors_cellstr = cellstr(WIOD_country_IO(3,:));   % Sector codes in IO data (column headers)

% Identify data rows (rows with valid NACE sector codes, excluding header rows)
data_rows_mask = ismember(NA_sectors_cellstr_full, NACE_55);

% Extract only data rows for cellstr arrays and numeric conversion
NA_sectors_cellstr = NA_sectors_cellstr_full(data_rows_mask);
NA_countries_cellstr = NA_countries_cellstr_full(data_rows_mask);

% Pre-convert cell data to numeric matrices ONCE (only data rows)
NA_data_numeric = cell2mat(WIOD_country_NA(data_rows_mask, 5:end));
IO_data_numeric = cell2mat(WIOD_country_IO(data_rows_mask, 5:end));

% Create logical arrays for rows (NA data) and columns (IO data)
rows_NACE = cell(1, length(sector_names));
columns_NACE = cell(1, length(sector_names));

for i = 1:length(sector_names)
    sector_indices = sector_map.(sector_names{i});
    rows_NACE{i} = ismember(NA_sectors_cellstr, NACE_55(sector_indices));
    columns_NACE{i} = ismember(IO_sectors_cellstr, NACE_55(sector_indices));
end




for i=1:length(ISO2)
    IO_rows.(ISO2{i}) = zeros(length(NACE),length(NACE_55));
    IO.(ISO2{i}) = zeros(length(NACE),length(NACE));
    NA.(ISO2{i}) = zeros(length(NACE),5);

    % Pre-compute country mask once per country (not per sector)
    country_mask = ismember(NA_countries_cellstr, ISO2(i));

    for j=1:length(rows_NACE(1,:))
    % Logicals for country-sector combination in rows
    row_mask = country_mask & rows_NACE{1,j};
    rows.(ISO2{i}).(NACE(j)) = row_mask;

    % Aggregate NA using pre-converted numeric matrix
    NA.(ISO2{i})(j,:) = sum(NA_data_numeric(row_mask, :), 1);

    % Aggregate IO using pre-converted numeric matrix
    IO_rows.(ISO2{i})(j,:) = sum(IO_data_numeric(row_mask, :), 1);
    end
    % over columns
    for j=1:length(rows_NACE(1,:))
    IO.(ISO2{i})(:,j) = sum(IO_rows.(ISO2{i})(:,columns_NACE{1,j}(5:end)),2);
    end

    % Convert into table format
    NA.(ISO2{i}) = array2table(NA.(ISO2{i}), 'VariableNames',{'Con_HH', 'Con_NP', 'CON_Gov', 'GFCF', 'Inv'}, 'RowNames', NACE);
    IO.(ISO2{i}) = array2table(IO.(ISO2{i}), 'VariableNames', NACE, 'RowNames', NACE);

end



end


