function CO2 = extract_and_aggregate_emissions(CO2, ISO_a, ISO_b, ISO_c, ISO_d, ISO_regions, all_sectors, NACE, NACE_available)
%EXTRACT_AND_AGGREGATE_EMISSIONS Aggregate CO2 emissions by region and sector
%
%   CO2 = EXTRACT_AND_AGGREGATE_EMISSIONS(CO2, ISO_a, ISO_b, ISO_c, ISO_d,
%   ISO_regions, all_sectors, NACE, NACE_available) aggregates CO2 emissions
%   from individual countries into regional totals and optionally aggregates
%   to custom sector groupings.
%
% INPUTS:
%   CO2           - Structure with CO2 emissions tables by country code
%                   Each table: 57 sectors x 17 years (2000-2016)
%   ISO_a         - Cell array of ISO codes for Region A
%   ISO_b         - Cell array of ISO codes for Region B
%   ISO_c         - Cell array of ISO codes for Region C
%   ISO_d         - Cell array of ISO codes for Region D
%   ISO_regions   - String array of region names (e.g., ["Reg_a", "Reg_b", ...])
%   all_sectors   - Flag: 1 = keep all 54 NACE sectors, 0 = aggregate to NACE
%   NACE          - String array of target sector codes (if all_sectors=0)
%   NACE_available - String array of available sector aggregations
%
% OUTPUTS:
%   CO2           - Structure with aggregated CO2 emissions tables by region
%                   Each table: n_sectors x 17 years, including FC_HH (household consumption)
%
% NOTES:
%   - Excludes 'ROW' countries (no data in WIOD environmental accounts)
%   - Last row (FC_HH) contains household final consumption emissions
%   - Uses get_sector_mappings() for sector aggregation when all_sectors=0
%   - Data source: WIOD Environmental Accounts (2000-2016)
%
% See also: MAIN_CLIMATE, GET_SECTOR_MAPPINGS

%% Input validation
if ~isstruct(CO2)
    error('extract_and_aggregate_emissions:InvalidCO2', ...
        'CO2 must be a structure with emissions tables by country code');
end

if ~iscell(ISO_a)
    error('extract_and_aggregate_emissions:InvalidISOa', ...
        'ISO_a must be a cell array of country codes for Region A');
end

if ~iscell(ISO_b)
    error('extract_and_aggregate_emissions:InvalidISOb', ...
        'ISO_b must be a cell array of country codes for Region B');
end

if ~iscell(ISO_c)
    error('extract_and_aggregate_emissions:InvalidISOc', ...
        'ISO_c must be a cell array of country codes for Region C');
end

if ~iscell(ISO_d)
    error('extract_and_aggregate_emissions:InvalidISOd', ...
        'ISO_d must be a cell array of country codes for Region D');
end

if isempty(ISO_regions)
    error('extract_and_aggregate_emissions:EmptyRegions', ...
        'ISO_regions cannot be empty');
end

if ~isscalar(all_sectors) || ~(all_sectors == 0 || all_sectors == 1)
    error('extract_and_aggregate_emissions:InvalidAllSectors', ...
        'all_sectors must be 0 (aggregate sectors) or 1 (all 54 sectors)');
end

if isempty(NACE)
    error('extract_and_aggregate_emissions:EmptyNACE', ...
        'NACE sector list cannot be empty');
end

%% Constants
% Number of sectors in full NACE classification (56 sectors + FC_HH for household consumption)
NUM_SECTORS_FULL = 57;
% Number of years in CO2 emissions data (2000-2016)
NUM_YEARS_CO2 = 17;
% Year labels for table columns
YEAR_LABELS = ["y00", "y01", "y02", "y03", "y04", "y05", "y06", "y07", "y08", "y09", "y10", "y11", "y12", "y13", "y14", "y15", "y16"];

%% Exclude 'ROW' since not included in SEA (GO from SEA appears in denominator of emissions intensity)
if ismember('ROW', ISO_a)
ISO_a = ISO_a(1:end-1);
end

if ismember('ROW', ISO_b)
ISO_b = ISO_b(1:end-1);
end

if ismember('ROW', ISO_c)
ISO_c = ISO_c(1:end-1);
end

if ismember('ROW', ISO_d)
ISO_d = ISO_d(1:end-1);
end


%% Aggregate over countries

for r=1:length(ISO_regions)
CO2.(ISO_regions(r)) = zeros(NUM_SECTORS_FULL, NUM_YEARS_CO2);  % Initialize
end

for c=1:length(ISO_a)
CO2.Reg_a = CO2.Reg_a + table2array(CO2.(ISO_a{c}));
end

for c=1:length(ISO_b)
CO2.Reg_b = CO2.Reg_b + table2array(CO2.(ISO_b{c}));
end

for c=1:length(ISO_c)
CO2.Reg_c = CO2.Reg_c + table2array(CO2.(ISO_c{c}));
end

for c=1:length(ISO_d)
CO2.Reg_d = CO2.Reg_d + table2array(CO2.(ISO_d{c}));
end

% Transform into tables
for r=1:length(ISO_regions)
CO2.(ISO_regions(r)) = array2table(CO2.(ISO_regions(r)), 'RowNames', CO2.(ISO_a{1}).Properties.RowNames, 'VariableNames', YEAR_LABELS);
end


%% Aggregate sectors
if all_sectors == 0

% Get sector mappings from shared utility function
[sector_map, NACE_55] = get_sector_mappings();

NACE_exclude = NACE_available(~ismember(NACE_available, NACE));

% Define the order of sector aggregations (must match extract_WIOD_sectors.m)
sector_names = {'A', 'B', 'C', 'MF_wo_C19', 'D', 'E', 'B_C_D_E', 'B_D_E', 'B_C19_D', 'B_C19', ...
                'F', 'E_F', 'G_H_I', 'H', 'J', 'K', 'L', 'M_N', 'O_P_Q', 'R_S', 'Serv', 'E_F_Serv_wo_H'};

% Create logical arrays for rows
rows_NACE = cell(1, length(sector_names));
for i = 1:length(sector_names)
    sector_indices = sector_map.(sector_names{i});
    rows_NACE{i} = ismember(CO2.(ISO_regions{1}).Properties.RowNames, NACE_55(sector_indices));
end


for r=1:length(ISO_regions)
    Co2_mat = zeros(length(rows_NACE)+1, size( CO2.(ISO_regions{r}),2));
    for i=1:length(rows_NACE)
    Co2_mat(i,:) = sum(table2array(CO2.(ISO_regions{r})(rows_NACE{i},:)),1);
    end

   Co2_mat(end,:) = table2array(CO2.(ISO_regions{r})(end,:));

   CO2.(ISO_regions{r}) = Co2_mat;
   CO2.(ISO_regions{r}) = array2table(CO2.(ISO_regions{r}), 'RowNames', [NACE_available, 'FC_HH'], 'VariableNames', YEAR_LABELS);

   % Drop NACE_exclude sectors
   for i=1:length(NACE_exclude)
   CO2.(ISO_regions{r})(NACE_exclude(i),:) =[];
   end


end
end


end
