function [sector_map, NACE_55] = get_sector_mappings()
% GET_SECTOR_MAPPINGS Create index mappings for sector aggregations
%
% Returns struct with fields for each aggregate sector containing
% index ranges for the constituent NACE sectors in the NACE_55 array.
%
% OUTPUTS:
%   sector_map - Struct with index ranges for each aggregate sector
%   NACE_55    - String array of all 54 NACE sector codes (U excluded)
%
% SECTOR MAPPINGS:
%   A           - Agriculture, forestry and fishing (indices 1:3)
%   B           - Mining and quarrying (index 4)
%   C           - Manufacturing (indices 5:23)
%   MF_wo_C19   - Manufacturing without C19 (indices 4:9, 11:26)
%   D           - Electricity, gas, steam and air conditioning (index 24)
%   E           - Water supply, sewerage, waste management (indices 25:26)
%   B_C_D_E     - All industrial sectors (indices 4:26)
%   B_D_E       - Industrial without manufacturing (indices 4, 24:26)
%   B_C19_D     - Mining, mineral oil, electricity (indices 4, 10, 24)
%   B_C19       - Mining and mineral oil (indices 4, 10)
%   F           - Construction (index 27)
%   E_F         - Water, waste and construction (indices 25:27)
%   G_H_I       - Trade, transport, accommodation (indices 28:36)
%   H           - Transportation and storage (indices 31:35)
%   J           - Information and communication (indices 37:40)
%   K           - Financial service activities (indices 41:43)
%   L           - Real estate activities (index 44)
%   M_N         - Professional, administrative services (indices 45:50)
%   O_P_Q       - Public administration, education, health (indices 51:53)
%   R_S         - Arts, entertainment, other services (index 54)
%   Serv        - All service sectors (indices 28:54)
%   E_F_Serv_wo_H - E, F and services without transport (indices 25:27, 28:30, 36:54)
%
% See also: EXTRACT_WIOD_SECTORS, EXTRACT_SEA_SECTORS, EXTRACT_AND_AGGREGATE_EMISSIONS

    % Define the NACE_55 sector classification (54 sectors - U excluded)
    NACE_55 = {'A01', 'A02', 'A03', 'B', 'C10-C12', 'C13-C15', 'C16', 'C17', 'C18', 'C19', 'C20', 'C21',...
        'C22', 'C23', 'C24', 'C25', 'C26', 'C27', 'C28', 'C29', 'C30', 'C31_C32', 'C33',...
        'D35', 'E36', 'E37-E39', 'F', 'G45', 'G46', 'G47', 'H49', 'H50', 'H51', 'H52', 'H53',...
        'I', 'J58', 'J59_J60', 'J61', 'J62_J63', 'K64', 'K65', 'K66', 'L68',...
        'M69_M70', 'M71', 'M72', 'M73', 'M74_M75', 'N', 'O84', 'P85', 'Q', 'R_S', 'T'};
    NACE_55 = string(NACE_55);

    % Define index mappings for aggregate sectors
    sector_map.A = 1:3;
    sector_map.B = 4;
    sector_map.C = 5:23;
    sector_map.MF_wo_C19 = [4:9, 11:26];
    sector_map.D = 24;
    sector_map.E = 25:26;
    sector_map.B_C_D_E = 4:26;
    sector_map.B_D_E = [4, 24:26];
    sector_map.B_C19_D = [4, 10, 24];
    sector_map.B_C19 = [4, 10];
    sector_map.F = 27;
    sector_map.E_F = 25:27;
    sector_map.G_H_I = 28:36;
    sector_map.H = 31:35;
    sector_map.J = 37:40;
    sector_map.K = 41:43;
    sector_map.L = 44;
    sector_map.M_N = 45:50;
    sector_map.O_P_Q = 51:53;
    sector_map.R_S = 54;
    sector_map.Serv = 28:54;
    sector_map.E_F_Serv_wo_H = [25:27, 28:30, 36:54];
end
