%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                       %
%    THE ENVIRONMENTAL MULTI-SECTOR DSGE MODEL EMuSe                    %
%                                                                       %
%    EMuSe has been developed by the                                    %
%    Directorate General Economics of the Deutsche Bundesbank.          %
%                                                                       %
%    Authors:                                                           %
%    Natascha Hinterlang                                                %
%    Anika Martin                                                       %
%    Oke Röhe                                                           %
%    Nikolai Stähler                                                    %
%    Johannes Strobel                                                   %
%                                                                       %
%    Contact: emuse@bundesbank.de
%
%    Last update: 17.07.2024
%                                                                       %
%    The authors are grateful to their colleagues in DG Economics,      %
%    DG Financial Stability and the Research Centre of Deutsche         %
%    Bundesbank, the Working Group on Econometric Modelling (WGEM) and  %
%    the Working Group on Forecasting (WGF) of the European System of   %
%    Central Banks (ESCB) as well as the members of the informal        %
%    network of modelling experts of the G7 Climate Change Mitigation   %
%    Working Group for their helpful discussions and valuable input     %
%    during the development of EMuSe.                                   %
%                                                                       %
%    If you use the EMuSe model, please cite                            %
%    Natascha Hinterlang, Anika Martin, Oke Röhe,                       %
%    Nikolai Stähler and Johannes Strobel (2023),                       %
%    The Environmental Multi-Sector DSGE model EMuSe:                   %
%    A technical documentation,                                         %
%    Deutsche Bundesbank Technical Paper, No. 03/2023.                  %
%                                                                       %
%                                                                       %
% Access to EMuSe is only granted on the basis set out in the           %
% accompanying End User License Agreement (EULA).                       %
% Downloading, installing or using the EMuSe model implies acceptance   %
% of the EULA. If you do not agree to be bound by these terms,          %
% do not download, install or use the related Software and              %
% documentation. As further outlined in the EULA the Software           %
% is provided "as is", without any representation or warranty of any    %
% kind either express or implied.                                       %
%                                                                       %
% The Software and Derived Work from the Software may only be           %
% distributed and communicated together with a copy of the EULA and the %
% aforementioned notice.                                                %
%                                                                       %
%                                                                       %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


% Calibrate EMuSe model using WIOD database and WIOD environmental accounts


% Housekeeping
clear
clc

%% 1) Load configuration
% Edit config_EMuSe_calibration.m to customize settings
config = config_EMuSe_calibration();

% Add data and routines folder to path using config settin
addpath(genpath(config.data_dir));
addpath(genpath(config.func_dir));

% Extract commonly used settings for convenience
years = config.years;
year_codes = config.year_codes;
years_available_WIOD = config.year_codes_available;
all_countries = config.all_countries;
all_sectors = config.all_sectors;

% Verify WIOT data files exist
WIOT_mat_files = dir(fullfile(config.wiot_dir, '*.mat'));
if isempty(WIOT_mat_files)
    error('EMuSe:DataNotFound', ...
        'No WIOT .mat files found in %s.\nPlease ensure the WIOD data files are in the correct directory.', ...
        config.wiot_dir);
end



%% 2) Load data
% All file paths use fullfile() for cross-platform compatibility

try
    load(config.sea_file);   % Load SEA data
catch ME
    error('EMuSe:LoadError', ...
        'Failed to load %s: %s\nEnsure the file exists.', config.sea_file, ME.message);
end

try
    load(config.nace_file);  % Load description of sectors
catch ME
    error('EMuSe:LoadError', ...
        'Failed to load %s: %s\nEnsure the file exists.', config.nace_file, ME.message);
end

% Load WIOD mat-files for chosen years
% Initialize
for i=1:length(years)
    WIOD_all.(year_codes{i}) = struct();
end

index = zeros(size(years));
for i=1:length(years)
    index(i) = find(ismember(years_available_WIOD, year_codes(i))==1);
    if isempty(index(i)) || index(i) > length(WIOT_mat_files)
        error('EMuSe:YearNotFound', ...
            'WIOT data for year %s not found. Available years: %s', ...
            year_codes{i}, strjoin(years_available_WIOD, ', '));
    end
    % Extract matname for backward compatibility
    [~, matname, ~] = fileparts(WIOT_mat_files(index(i)).name);
    % Use fullfile for cross-platform path construction
    wiot_file = fullfile(config.wiot_dir, WIOT_mat_files(index(i)).name);
    try
        load(wiot_file);
        WIOD_all.(year_codes{i}) = WIOD;
    catch ME
        error('EMuSe:LoadError', ...
            'Failed to load WIOT file %s for year %s: %s', ...
            wiot_file, year_codes{i}, ME.message);
    end
end

ISO_SEA = unique(WIOD_SEA_dat.country); % Countries included in SEA
ISO_WIO = unique(cellstr(WIOD_all.(year_codes{1})(:,3)));    % Countries included in World Input-Output Tables (WIOT)
ISO_only_in_WIO = ISO_WIO(~ismember(ISO_WIO, ISO_SEA)); % 'ROW' only included in WIOT, not in SEA
ISO_all = ISO_WIO(ismember(ISO_WIO, ISO_SEA));


% Load exchange rates
try
    load(config.exchange_file);
catch ME
    error('EMuSe:LoadError', ...
        'Failed to load %s: %s\nEnsure the file exists.', config.exchange_file, ME.message);
end
Exchange.Properties.RowNames = ISO_all ;  % Add ISO names to exchange rate table


%% 3) Set up regions from configuration
% Region settings are defined in config_EMuSe_calibration.m

if all_countries == 1
    ISO_all = [ISO_all; 'ROW']';   % Needed to extract data for ALL countries
end

% Export predefined country groups to workspace (for backward compatibility)
ISO_EU_28 = config.ISO_EU_28;
ISO_EA_19 = config.ISO_EA_19;

% Load region definitions from config
ISO_a = config.ISO_a;
ISO_b = config.ISO_b;
ISO_c = config.ISO_c;
ISO_d = config.ISO_d;
ISO_regions = config.ISO_regions;
nr_regions = config.nr_regions;

% Auto-populate empty regions with remaining countries (ROW logic)
% Also define ISO_ROW for backward compatibility
if isempty(ISO_b) && nr_regions >= 2
    ISO_ROW = ISO_all(~ismember(ISO_all, [ISO_a, ISO_c, ISO_d]));
    ISO_b = ISO_ROW;
elseif isempty(ISO_c) && nr_regions >= 3
    ISO_ROW = ISO_all(~ismember(ISO_all, [ISO_a, ISO_b, ISO_d]));
    ISO_c = ISO_ROW;
elseif isempty(ISO_d) && nr_regions >= 4
    ISO_ROW = ISO_all(~ismember(ISO_all, [ISO_a, ISO_b, ISO_c]));
    ISO_d = ISO_ROW;
else
    % ISO_ROW = remaining countries not in any region
    ISO_ROW = ISO_all(~ismember(ISO_all, [ISO_a, ISO_b, ISO_c, ISO_d]));
end

% Combine all countries
ISO = [ISO_a, ISO_b, ISO_c, ISO_d];
ISO2 = ISO;
ISO_all = ISO;
nr_countries = length(ISO);


%% 4) Set up sectors from configuration
% Sector settings are defined in config_EMuSe_calibration.m

if all_sectors == 1
    % Use all 54 NACE sectors
    NACE = config.NACE_54;
    NACE_available = NACE;
else
    % Use custom sector aggregates
    NACE_available = config.NACE_available;
    NACE = config.NACE;
    NACE_exclude = config.NACE_exclude;
end

nr_sectors = config.nr_sectors;


%% 5) Extract data from Socio-Economic Accounts and compute parameters for production function and labor and capital agencies

if all_sectors == 1

    for c = 1:nr_countries

      % Extract
      Data.(ISO{c}) = struct();

     [Data.(ISO{c}).A01, Data.(ISO{c}).A02, Data.(ISO{c}).A03,...
      Data.(ISO{c}).B, Data.(ISO{c}).C10_C12, Data.(ISO{c}).C13_C15,...
      Data.(ISO{c}).C16, Data.(ISO{c}).C17, Data.(ISO{c}).C18,...
      Data.(ISO{c}).C19, Data.(ISO{c}).C20, Data.(ISO{c}).C21,...
      Data.(ISO{c}).C22, Data.(ISO{c}).C23, Data.(ISO{c}).C24,...
      Data.(ISO{c}).C25, Data.(ISO{c}).C26, Data.(ISO{c}).C27,...
      Data.(ISO{c}).C28, Data.(ISO{c}).C29, Data.(ISO{c}).C30,...
      Data.(ISO{c}).C31_C32, Data.(ISO{c}).C33, Data.(ISO{c}).D35, Data.(ISO{c}).E36,...
      Data.(ISO{c}).E37_E39, Data.(ISO{c}).F, Data.(ISO{c}).G45,...
      Data.(ISO{c}).G46, Data.(ISO{c}).G47, Data.(ISO{c}).H49, Data.(ISO{c}).H50,...
      Data.(ISO{c}).H51, Data.(ISO{c}).H52, Data.(ISO{c}).H53,...
      Data.(ISO{c}).I, Data.(ISO{c}).J58, Data.(ISO{c}).J59_J60,...
      Data.(ISO{c}).J61, Data.(ISO{c}).J62_J63, Data.(ISO{c}).K64,...
      Data.(ISO{c}).K65, Data.(ISO{c}).K66, Data.(ISO{c}).L68,...
      Data.(ISO{c}).M69_M70, Data.(ISO{c}).M71, Data.(ISO{c}).M72, ...
      Data.(ISO{c}).M73, Data.(ISO{c}).M74_M75, Data.(ISO{c}).N,...
      Data.(ISO{c}).O84, Data.(ISO{c}).P85, Data.(ISO{c}).Q,...
      Data.(ISO{c}).R_S] = extract_SEA_all_sectors(WIOD_SEA_dat,ISO(c));

    end

    try
        Calibration_SEA = aggregate_countries_SEA(ISO_a, ISO_b, ISO_c, ISO_d, ISO_regions, NACE, Data, Exchange);
    catch ME
        error('EMuSe:AggregationError', ...
            'Error in SEA aggregation for regions [%s]: %s', ...
            strjoin(cellstr(ISO_regions), ', '), ME.message);
    end

else

    for c = 1:nr_countries
      Data.(ISO{c}) = struct();

     [Data.(ISO{c}).A, Data.(ISO{c}).B, Data.(ISO{c}).C,...
      Data.(ISO{c}).MF_wo_C19, Data.(ISO{c}).D, Data.(ISO{c}).E, ...
      Data.(ISO{c}).B_C_D_E, Data.(ISO{c}).B_D_E, Data.(ISO{c}).B_C19_D,  ...
      Data.(ISO{c}).B_C19, Data.(ISO{c}).F, Data.(ISO{c}).E_F, ...
      Data.(ISO{c}).G_H_I, Data.(ISO{c}).H, Data.(ISO{c}).J, Data.(ISO{c}).K, ...
      Data.(ISO{c}).L, Data.(ISO{c}).M_N, Data.(ISO{c}).O_P_Q, ...
      Data.(ISO{c}).R_S, Data.(ISO{c}).Serv, Data.(ISO{c}).E_F_Serv_wo_H] = extract_SEA_sectors(WIOD_SEA_dat,ISO(c));

      % Drop NACE_exclude sectors
      Data.(ISO{c}) =rmfield(Data.(ISO{c}), NACE_exclude);

    end

    try
        Calibration_SEA = aggregate_countries_SEA(ISO_a, ISO_b, ISO_c, ISO_d, ISO_regions, NACE, Data, Exchange);
    catch ME
        error('EMuSe:AggregationError', ...
            'Error in SEA aggregation for regions [%s]: %s', ...
            strjoin(cellstr(ISO_regions), ', '), ME.message);
    end

end

Calibration= struct(); % Initialize
Calibration.SEA = Calibration_SEA;


%% 6) Extract data from World Input-Output Tables, aggregate countries to regions and compute parameters

Calibration.WIOD=struct(); % Initialize

for i=1:length(years)
Calibration.WIOD.(year_codes{i}) =struct();
end

for i=1:length(years)

if all_sectors ==1

    for c = 1:nr_countries
    % Exract IO and NA data
      [Data.(ISO{c}).NA, Data.(ISO{c}).IO] = extract_WIOD_all_sectors(WIOD_all.(year_codes{i}), ISO(c), ISO2, NACE);

    % Set negative GFCF values to zero since otherwise negative Psi_I values possible
    Data.(ISO{c}).NA.(ISO{c}) = table2array(Data.(ISO{c}).NA.(ISO{c}));
    rows_negative = (Data.(ISO{c}).NA.(ISO{c})(:,4)) < 0;
    Data.(ISO{c}).NA.(ISO{c})(rows_negative,4) =0;
    Data.(ISO{c}).NA.(ISO{c}) = array2table(Data.(ISO{c}).NA.(ISO{c}), 'VariableNames',{'Psi_C', 'Psi_Con_NP', 'Psi_G', 'Psi_I', 'Psi_Inventories'}, 'RowNames', NACE);

    end
    try
        Calibration_WIOD = aggregate_countries_WIOD(ISO_all, ISO_a, ISO_b, ISO_c, ISO_d, ISO_regions, NACE, Data);
    catch ME
        error('EMuSe:AggregationError', ...
            'Error in WIOD aggregation for year %s, regions [%s]: %s', ...
            year_codes{i}, strjoin(cellstr(ISO_regions), ', '), ME.message);
    end

else

      for c = 1:nr_countries
      % Exract IO and NA data
      [Data.(ISO{c}).NA, Data.(ISO{c}).IO] = extract_WIOD_sectors(WIOD_all.(year_codes{i}), ISO(c), ISO2, NACE_available);

      % Drop NACE_exclude sectors
      for c2 = 1:nr_countries
      Data.(ISO{c}).NA.(ISO{c2})(NACE_exclude,:) = [];
      Data.(ISO{c}).IO.(ISO{c2})(NACE_exclude,:) = [];
      Data.(ISO{c}).IO.(ISO{c2})(:,NACE_exclude) = [];
      end

      % Set negative GFCF values to zero since otherwise negative Psi_I values
      Data.(ISO{c}).NA.(ISO{c}) = table2array(Data.(ISO{c}).NA.(ISO{c}));
      rows_negative = (Data.(ISO{c}).NA.(ISO{c})(:,4)) < 0;
      Data.(ISO{c}).NA.(ISO{c})(rows_negative,4) =0;
      Data.(ISO{c}).NA.(ISO{c}) = array2table(Data.(ISO{c}).NA.(ISO{c}), 'VariableNames',{'Psi_C', 'Psi_Con_NP', 'Psi_G', 'Psi_I', 'Psi_Inventories'}, 'RowNames', NACE);

      end

      try
          Calibration_WIOD = aggregate_countries_WIOD(ISO_all, ISO_a, ISO_b, ISO_c, ISO_d, ISO_regions, NACE, Data);
      catch ME
          error('EMuSe:AggregationError', ...
              'Error in WIOD aggregation for year %s, regions [%s]: %s', ...
              year_codes{i}, strjoin(cellstr(ISO_regions), ', '), ME.message);
      end

end
Calibration.WIOD.(year_codes{i}) = Calibration_WIOD;
end

Calibration.Country_list = {ISO_a ;ISO_b; ISO_c; ISO_d};    % Add country ISOs to have an overview of which country is included in which region


%% 7) Compute means over years
Calibration = compute_wiod_means(Calibration, NACE, ISO_regions, year_codes);


%% 8) Compute emission intensities and add them to calibration mat file
% Load CO2 emissions data
try
    load(config.co2_file);
catch ME
    error('EMuSe:LoadError', ...
        'Failed to load %s: %s\nEnsure the file exists.', config.co2_file, ME.message);
end

% Extract and aggregate emissions (updates CO2 with region fields for backward compatibility)
CO2 = extract_and_aggregate_emissions(CO2, ISO_a, ISO_b, ISO_c, ISO_d, ...
    ISO_regions, all_sectors, NACE, NACE_available);

Calibration = compute_emission_intensities(Calibration, CO2, ...
    ISO_a, ISO_b, ISO_c, ISO_d, ISO_regions, NACE, NACE_available, all_sectors);


