function config = config_EMuSe_calibration()
%CONFIG_EMUSE_CALIBRATION Configuration settings for EMuSe Calibration Toolkit
%
%   config = CONFIG_EMUSE_CALIBRATION() returns a structure containing all
%   user-configurable settings for the EMuSe calibration process.
%
% USAGE:
%   Edit this file to customize the calibration settings, then run
%   Main_EMuSe_Calibration_Toolkit.m which will load these settings.
%
% OUTPUT:
%   config - Structure with the following fields:
%     .years            - Numeric array of years to process (e.g., 2005 or 2000:2014)
%     .all_countries    - Include all countries including ROW (1=yes, 0=no)
%     .all_sectors      - Use all 54 NACE sectors (1=yes, 0=custom aggregates)
%     .ISO_a/b/c/d      - Cell arrays of country codes for each region
%     .ISO_regions      - String array of region names
%     .NACE             - String array of selected sector codes
%     .data_dir         - Path to data directory
%     .func_dir         - Path to functions directory
%     .output_dir       - Path to output directory
%
% EXAMPLES:
%   % Default configuration (EU-28 vs Rest of World)
%   config = config_EMuSe_calibration();
%
%   % To customize, edit this file or create a modified copy
%
% See also: MAIN_EMUSE_CALIBRATION_TOOLKIT

%% =========================================================================
%  YEAR SETTINGS
%  =========================================================================

% Available years in WIOD database (DO NOT CHANGE)
config.years_available = 2000:2014;
config.year_codes_available = {'y00', 'y01', 'y02', 'y03', 'y04', 'y05', ...
    'y06', 'y07', 'y08', 'y09', 'y10', 'y11', 'y12', 'y13', 'y14'};

% SELECT YEARS TO PROCESS
% Single year example:
config.years = 2005;

% Multiple years example (uncomment to use):
% config.years = 2000:2014;

% Generate year codes from selected years
config.year_codes = cell(1, length(config.years));
for i = 1:length(config.years)
    config.year_codes{i} = sprintf('y%02d', mod(config.years(i), 100));
end

%% =========================================================================
%  COUNTRY/REGION SETTINGS
%  =========================================================================

% Include all countries including 'ROW' (Rest of World from WIOT)
% Set to 1 when calibrating the rest of the world, 0 otherwise
config.all_countries = 1;

% PREDEFINED COUNTRY GROUPS (for convenience)
% Define as local variables first, then assign to config
ISO_EU_28 = {'AUT', 'BEL', 'BGR', 'CYP', 'CZE', 'DEU', 'DNK', 'ESP', ...
    'EST', 'FIN', 'FRA', 'GBR', 'GRC', 'HRV', 'HUN', 'IRL', 'ITA', 'LTU', ...
    'LUX', 'LVA', 'MLT', 'NLD', 'POL', 'PRT', 'ROU', 'SVK', 'SVN', 'SWE'};

ISO_EA_19 = {'AUT', 'BEL', 'CYP', 'DEU', 'ESP', 'EST', 'FIN', 'FRA', ...
    'GRC', 'IRL', 'ITA', 'LTU', 'LUX', 'LVA', 'MLT', 'NLD', 'PRT', 'SVK', 'SVN'};

% Store in config for reference
config.ISO_EU_28 = ISO_EU_28;
config.ISO_EA_19 = ISO_EA_19;

% DEFINE REGIONS
% The toolkit supports up to 4 regions (Reg_a, Reg_b, Reg_c, Reg_d)
% Leave empty {} for unused regions

% CURRENT CONFIGURATION: EU-28 vs Rest of World (2 regions)
config.ISO_a = ISO_EU_28;  % Region A: EU-28
config.ISO_b = {};                 % Region B: Will be set to ROW automatically
config.ISO_c = {};                 % Region C: Empty
config.ISO_d = {};                 % Region D: Empty

% Active regions (only include regions that have countries)
config.ISO_regions = ["Reg_a", "Reg_b"];

% ALTERNATIVE CONFIGURATIONS (uncomment to use):

% --- Germany vs Rest of EA vs Rest of World (3 regions) ---
% config.ISO_a = {'DEU'};
% config.ISO_b = ISO_EA_19(~ismember(ISO_EA_19, {'DEU'}));
% config.ISO_c = {};  % Will be set to remaining countries
% config.ISO_d = {};
% config.ISO_regions = ["Reg_a", "Reg_b", "Reg_c"];

% --- Single region (e.g., Germany only) ---
% config.ISO_a = {'DEU'};
% config.ISO_b = {};
% config.ISO_c = {};
% config.ISO_d = {};
% config.ISO_regions = ["Reg_a"];

%% =========================================================================
%  SECTOR SETTINGS
%  =========================================================================

% Use all 54 NACE sectors at two-digit level (1) or custom aggregates (0)
config.all_sectors = 0;

% ALL 54 NACE SECTORS (used when all_sectors = 1)
config.NACE_54 = ["A01", "A02", "A03", "B", "C10_C12", "C13_C15", "C16", ...
    "C17", "C18", "C19", "C20", "C21", "C22", "C23", "C24", "C25", "C26", ...
    "C27", "C28", "C29", "C30", "C31_C32", "C33", "D35", "E36", "E37_E39", ...
    "F", "G45", "G46", "G47", "H49", "H50", "H51", "H52", "H53", "I", ...
    "J58", "J59_J60", "J61", "J62_J63", "K64", "K65", "K66", "L68", ...
    "M69_M70", "M71", "M72", "M73", "M74_M75", "N", "O84", "P85", "Q", "R_S"];

% AVAILABLE AGGREGATE SECTORS (used when all_sectors = 0)
config.NACE_available = ["A", "B", "C", "MF_wo_C19", "D", "E", "B_C_D_E", ...
    "B_D_E", "B_C19_D", "B_C19", "F", "E_F", "G_H_I", "H", "J", "K", "L", ...
    "M_N", "O_P_Q", "R_S", "Serv", "E_F_Serv_wo_H"];

% SELECT SECTORS TO USE (when all_sectors = 0)
% This is the broad 10-sector NACE structure used in EMuSe documentation
config.NACE = ["A", "B", "C", "D", "E", "F", "G_H_I", "J", "M_N", "R_S"];

% ALTERNATIVE SECTOR CONFIGURATIONS (uncomment to use):

% --- 6 sectors (very aggregated) ---
% config.NACE = ["A", "B_C_D_E", "F", "G_H_I", "J", "Serv"];

% --- Services focus ---
% config.NACE = ["A", "B", "C", "D", "E", "F", "G_H_I", "H", "J", "K", "L", "M_N", "O_P_Q", "R_S"];

%% =========================================================================
%  PATH SETTINGS
%  =========================================================================

% All paths are relative to EMuSe_Calibration_Toolkit folder
% Using fullfile() ensures cross-platform compatibility (Windows/Linux/macOS)

% Functions directory
config.functions_dir = 'Functions';

% Data directory
config.data_dir = 'Data';

% Routines directory
config.func_dir = 'Functions';

% WIOT subdirectory (World Input-Output Tables)
config.wiot_dir = fullfile(config.data_dir, 'WIOT');

% Specific data files
config.sea_file = fullfile(config.data_dir, 'WIOD_SEA_dat.mat');
config.nace_file = fullfile(config.data_dir, 'NACE_55_descr.mat');
config.exchange_file = fullfile(config.data_dir, 'WIOD_Exchange_rates.mat');
config.co2_file = fullfile(config.data_dir, 'CO2_emissions.mat');

% Output directory for calibration results
config.output_dir = 'Output';

%% =========================================================================
%  DERIVED SETTINGS (computed automatically, do not modify)
%  =========================================================================

% Sectors to exclude when using custom aggregates
if config.all_sectors == 0
    config.NACE_exclude = config.NACE_available(~ismember(config.NACE_available, config.NACE));
else
    config.NACE_exclude = [];
end

% Number of regions and sectors
config.nr_regions = length(config.ISO_regions);
config.nr_sectors = length(config.NACE);

end
