function [isEqual, differences] = deep_compare(data1, data2, varargin)
% Recursively compare two MATLAB variables
% [isEqual, diffs] = deep_compare(data1, data2)
% [isEqual, diffs] = deep_compare(data1, data2, path)

if nargin < 3
    path = 'root';
else
    path = varargin{1};
end

differences = {};
isEqual = true;

% OPTIMIZATION: Quick check if already identical
if isequaln(data1, data2)
    return;
end

% Handle empty
if isempty(data1) && isempty(data2)
    return;
end
if xor(isempty(data1), isempty(data2))
    differences{end+1} = sprintf('%s: Empty mismatch', path);
    isEqual = false;
    return;
end

% Check class
if ~strcmp(class(data1), class(data2))
    differences{end+1} = sprintf('%s: Class mismatch', path);
    isEqual = false;
    return;
end

% Handle by type
if isstruct(data1)
    [isEqual, differences] = cmp_struct(data1, data2, path);
elseif istable(data1)
    [isEqual, differences] = cmp_table(data1, data2, path);
elseif iscell(data1)
    [isEqual, differences] = cmp_cell(data1, data2, path);
elseif isnumeric(data1) || islogical(data1)
    [isEqual, differences] = cmp_numeric(data1, data2, path);
elseif ischar(data1) || isstring(data1)
    if ~isequal(data1, data2)
        differences{end+1} = sprintf('%s: String differs', path);
        isEqual = false;
    end
else
    if ~isequaln(data1, data2)
        differences{end+1} = sprintf('%s: Differs', path);
        isEqual = false;
    end
end
end

function [eq, diffs] = cmp_struct(s1, s2, p)
eq = true;
diffs = {};

% Quick check first
if isequaln(s1, s2)
    return;
end

f1 = fieldnames(s1);
f2 = fieldnames(s2);
if ~isequal(sort(f1), sort(f2))
    diffs{end+1} = sprintf('%s: Field mismatch', p);
    eq = false;
end
common = intersect(f1, f2);
for i = 1:length(common)
    fn = common{i};
    [e, d] = deep_compare(s1.(fn), s2.(fn), [p '.' fn]);
    if ~e
        eq = false;
        diffs = [diffs, d];
    end
end
end

function [eq, diffs] = cmp_table(t1, t2, p)
eq = true;
diffs = {};

% Quick check first
if isequaln(t1, t2)
    return;
end

if ~isequal(size(t1), size(t2))
    diffs{end+1} = sprintf('%s: Table size differs', p);
    eq = false;
    return;
end
try
    a1 = table2array(t1);
    a2 = table2array(t2);
    if isnumeric(a1)
        [e, d] = cmp_numeric(a1, a2, p);
        if ~e
            eq = false;
            diffs = [diffs, d];
        end
    else
        if ~isequaln(a1, a2)
            diffs{end+1} = sprintf('%s: Table data differs', p);
            eq = false;
        end
    end
catch
    if ~isequaln(t1, t2)
        diffs{end+1} = sprintf('%s: Table differs', p);
        eq = false;
    end
end
end

function [eq, diffs] = cmp_cell(c1, c2, p)
eq = true;
diffs = {};

% Quick check first
if isequaln(c1, c2)
    return;
end

if ~isequal(size(c1), size(c2))
    diffs{end+1} = sprintf('%s: Cell size differs', p);
    eq = false;
    return;
end
for i = 1:numel(c1)
    [e, d] = deep_compare(c1{i}, c2{i}, sprintf('%s{%d}', p, i));
    if ~e
        eq = false;
        diffs = [diffs, d];
    end
end
end

function [eq, diffs] = cmp_numeric(n1, n2, p)
eq = true;
diffs = {};
if ~isequal(size(n1), size(n2))
    diffs{end+1} = sprintf('%s: Size differs', p);
    eq = false;
    return;
end
if ~isequaln(n1, n2)
    diffs{end+1} = sprintf('%s: Numeric data differs', p);
    eq = false;
end
end
