%% Test Script - Deep Comparison Against Baseline
% Compares current workspace variables against testData.mat baseline
% Reports exact location of any differences in nested structures

fprintf('========================================\n');
fprintf('EMuSe Calibration - Baseline Test\n');
fprintf('========================================\n\n');

% Load baseline
fprintf('Loading baseline data (testData.mat)...\n');
oData = load('testData.mat');
allfields = fieldnames(oData);
fprintf('  Found %d variables to compare\n\n', length(allfields));

% Compare each variable
total_diffs = 0;
vars_with_diffs = 0;

% Variables to skip (filesystem metadata, not computation results)
skip_vars = {};

for i = 1:length(allfields)
    varname = allfields{i};

    % Skip excluded variables
    if ismember(varname, skip_vars)
        fprintf('Skipping: %s (filesystem metadata)\n', varname);
        continue;
    end

    if ~exist(varname, 'var')
        error('Variable %s does not exist in the workspace.', varname);
    end

    tmp = eval(varname);

    fprintf('Checking: %s ... ', varname);

    % Deep comparison
    [isEqual, differences] = deep_compare(oData.(varname), tmp, varname);

    if isEqual
        fprintf('✓ IDENTICAL\n');
    else
        fprintf('✗ DIFFERENCES FOUND (%d)\n', length(differences));
        vars_with_diffs = vars_with_diffs + 1;
        total_diffs = total_diffs + length(differences);

        % Print all differences for this variable
        for j = 1:length(differences)
            fprintf('  → %s\n', differences{j});
        end
        fprintf('\n');
    end
end

% Summary
fprintf('\n========================================\n');
fprintf('SUMMARY\n');
fprintf('========================================\n');
fprintf('Variables compared: %d\n', length(allfields));
fprintf('Variables with diffs: %d\n', vars_with_diffs);
fprintf('Total differences: %d\n', total_diffs);
fprintf('========================================\n\n');

if total_diffs > 0
    error('Test FAILED: %d differences found!', total_diffs);
else
    fprintf('✓✓✓ Test PASSED: All data identical to baseline!\n\n');
end
