function results = run_all_tests(options)
%RUN_ALL_TESTS Run all unit tests for EMuSe Calibration Toolkit
%
%   results = RUN_ALL_TESTS() runs all tests in the tests/ folder and
%   returns a TestResult array.
%
%   results = RUN_ALL_TESTS(options) allows customization:
%       options.Verbosity - 'Minimal', 'Normal', 'Detailed', 'Verbose' (default: 'Normal')
%       options.ReportCoverage - true/false to generate coverage report (default: false)
%       options.OutputFile - filename for JUnit XML report (default: '')
%
% EXAMPLES:
%   % Run all tests with default settings
%   results = run_all_tests();
%
%   % Run with detailed output
%   results = run_all_tests(struct('Verbosity', 'Detailed'));
%
%   % Generate JUnit XML report for CI
%   results = run_all_tests(struct('OutputFile', 'test-results.xml'));
%
% See also: runtests, matlab.unittest.TestSuite

    arguments
        options.Verbosity (1,1) string = "Normal"
        options.ReportCoverage (1,1) logical = false
        options.OutputFile (1,1) string = ""
    end

    % Get the directory containing this script
    testDir = fileparts(mfilename('fullpath'));
    rootDir = fileparts(testDir);

    fprintf('=================================================\n');
    fprintf('EMuSe Calibration Toolkit - Unit Test Suite\n');
    fprintf('=================================================\n');
    fprintf('Test directory: %s\n', testDir);
    fprintf('Running at: %s\n\n', datestr(now));

    % Add Functions folder to path temporarily
    funcPath = fullfile(rootDir, 'EMuSe_Calibration_Toolkit', 'Functions');
    addpath(funcPath);
    cleanupObj = onCleanup(@() rmpath(funcPath));

    % Create test suite from test folder
    suite = matlab.unittest.TestSuite.fromFolder(testDir);

    fprintf('Found %d test(s)\n\n', numel(suite));

    % Create test runner with appropriate verbosity
    import matlab.unittest.TestRunner
    import matlab.unittest.Verbosity

    % Map verbosity string to enum value
    switch lower(options.Verbosity)
        case "minimal"
            verbLevel = Verbosity.Terse;
        case "normal"
            verbLevel = Verbosity.Concise;
        case "detailed"
            verbLevel = Verbosity.Detailed;
        case "verbose"
            verbLevel = Verbosity.Verbose;
        otherwise
            verbLevel = Verbosity.Concise;
    end

    runner = TestRunner.withTextOutput('Verbosity', verbLevel);

    % Add JUnit XML plugin if output file specified
    if options.OutputFile ~= ""
        import matlab.unittest.plugins.XMLPlugin
        xmlPlugin = XMLPlugin.producingJUnitFormat(options.OutputFile);
        runner.addPlugin(xmlPlugin);
        fprintf('JUnit XML report will be saved to: %s\n\n', options.OutputFile);
    end

    % Add code coverage plugin if requested
    if options.ReportCoverage
        import matlab.unittest.plugins.CodeCoveragePlugin
        import matlab.unittest.plugins.codecoverage.CoverageReport

        coverageDir = fullfile(testDir, 'coverage');
        if ~exist(coverageDir, 'dir')
            mkdir(coverageDir);
        end

        % Get all .m files in Functions folder
        funcFiles = dir(fullfile(funcPath, '*.m'));
        sourceFiles = fullfile(funcPath, {funcFiles.name});

        if ~isempty(sourceFiles)
            coveragePlugin = CodeCoveragePlugin.forFile(sourceFiles, ...
                'Producing', CoverageReport(coverageDir));
            runner.addPlugin(coveragePlugin);
            fprintf('Coverage report will be saved to: %s\n\n', coverageDir);
        end
    end

    % Run the tests
    results = runner.run(suite);

    % Display summary
    fprintf('\n=================================================\n');
    fprintf('Test Summary\n');
    fprintf('=================================================\n');
    fprintf('Total:  %d\n', numel(results));
    fprintf('Passed: %d\n', sum([results.Passed]));
    fprintf('Failed: %d\n', sum([results.Failed]));
    fprintf('Incomplete: %d\n', sum([results.Incomplete]));
    fprintf('Duration: %.2f seconds\n', sum([results.Duration]));
    fprintf('=================================================\n');

    % Return non-zero exit if any tests failed (useful for CI)
    if any([results.Failed])
        fprintf('\n*** SOME TESTS FAILED ***\n');
    else
        fprintf('\n*** ALL TESTS PASSED ***\n');
    end
end
