classdef test_get_sector_mappings < matlab.unittest.TestCase
%TEST_GET_SECTOR_MAPPINGS Unit tests for get_sector_mappings function
%
%   Run with: runtests('test_get_sector_mappings')

    properties
        SectorMap
        NACE_55
    end

    methods(TestClassSetup)
        function setupOnce(testCase)
            % Add Functions folder to path
            funcPath = fullfile(fileparts(fileparts(mfilename('fullpath'))), ...
                'EMuSe_Calibration_Toolkit', 'Functions');
            addpath(funcPath);

            % Get the mappings once for all tests
            [testCase.SectorMap, testCase.NACE_55] = get_sector_mappings();
        end
    end

    methods(TestClassTeardown)
        function teardownOnce(testCase)
            % Remove Functions folder from path
            funcPath = fullfile(fileparts(fileparts(mfilename('fullpath'))), ...
                'EMuSe_Calibration_Toolkit', 'Functions');
            rmpath(funcPath);
        end
    end

    methods(Test)
        %% Output Structure Tests
        function testOutputTypes(testCase)
            % Verify output types
            testCase.verifyClass(testCase.SectorMap, 'struct');
            testCase.verifyClass(testCase.NACE_55, 'string');
        end

        function testNACE55Length(testCase)
            % NACE_55 should have 55 elements (54 sectors + T, U excluded in practice)
            testCase.verifyEqual(length(testCase.NACE_55), 55);
        end

        function testSectorMapFields(testCase)
            % Verify all expected sector aggregations exist
            expectedFields = {'A', 'B', 'C', 'MF_wo_C19', 'D', 'E', ...
                'B_C_D_E', 'B_D_E', 'B_C19_D', 'B_C19', 'F', 'E_F', ...
                'G_H_I', 'H', 'J', 'K', 'L', 'M_N', 'O_P_Q', 'R_S', ...
                'Serv', 'E_F_Serv_wo_H'};

            for i = 1:length(expectedFields)
                testCase.verifyTrue(isfield(testCase.SectorMap, expectedFields{i}), ...
                    sprintf('Missing field: %s', expectedFields{i}));
            end
        end

        %% Sector Mapping Tests
        function testAgricultureMapping(testCase)
            % A = Agriculture (indices 1:3)
            testCase.verifyEqual(testCase.SectorMap.A, 1:3);
            % Verify actual sector codes
            testCase.verifyEqual(testCase.NACE_55(testCase.SectorMap.A), ...
                ["A01", "A02", "A03"]);
        end

        function testMiningMapping(testCase)
            % B = Mining (index 4)
            testCase.verifyEqual(testCase.SectorMap.B, 4);
            testCase.verifyEqual(testCase.NACE_55(testCase.SectorMap.B), "B");
        end

        function testManufacturingMapping(testCase)
            % C = Manufacturing (indices 5:23)
            testCase.verifyEqual(testCase.SectorMap.C, 5:23);
            testCase.verifyEqual(length(testCase.SectorMap.C), 19);
        end

        function testServicesMapping(testCase)
            % Serv = All services (indices 28:54)
            testCase.verifyEqual(testCase.SectorMap.Serv, 28:54);
            testCase.verifyEqual(length(testCase.SectorMap.Serv), 27);
        end

        %% Index Validity Tests
        function testAllIndicesInRange(testCase)
            % All sector indices should be within 1:55
            fields = fieldnames(testCase.SectorMap);
            for i = 1:length(fields)
                indices = testCase.SectorMap.(fields{i});
                testCase.verifyGreaterThanOrEqual(min(indices), 1, ...
                    sprintf('Field %s has index < 1', fields{i}));
                testCase.verifyLessThanOrEqual(max(indices), length(testCase.NACE_55), ...
                    sprintf('Field %s has index > %d', fields{i}, length(testCase.NACE_55)));
            end
        end

        function testNoOverlapInMainSectors(testCase)
            % Main sectors (A, B, C, D, E, F, G_H_I, J, K, L, M_N, O_P_Q, R_S)
            % should not overlap and should cover indices 1:54
            mainSectors = {'A', 'B', 'C', 'D', 'E', 'F', 'G_H_I', 'J', 'K', 'L', 'M_N', 'O_P_Q', 'R_S'};
            allIndices = [];
            for i = 1:length(mainSectors)
                allIndices = [allIndices, testCase.SectorMap.(mainSectors{i})]; %#ok<AGROW>
            end
            % Check no duplicates
            testCase.verifyEqual(length(allIndices), length(unique(allIndices)), ...
                'Main sectors have overlapping indices');
            % Check coverage (should cover 1:54)
            testCase.verifyEqual(sort(allIndices), 1:54, ...
                'Main sectors do not cover all indices 1:54');
        end

        %% Composite Sector Tests
        function testB_C_D_E_Composition(testCase)
            % B_C_D_E should equal union of B, C, D, E
            expected = sort([testCase.SectorMap.B, testCase.SectorMap.C, ...
                testCase.SectorMap.D, testCase.SectorMap.E]);
            testCase.verifyEqual(sort(testCase.SectorMap.B_C_D_E), expected);
        end

        function testE_F_Composition(testCase)
            % E_F should equal union of E and F
            expected = sort([testCase.SectorMap.E, testCase.SectorMap.F]);
            testCase.verifyEqual(sort(testCase.SectorMap.E_F), expected);
        end

        function testMF_wo_C19_ExcludesC19(testCase)
            % MF_wo_C19 should not include index 10 (C19)
            testCase.verifyFalse(ismember(10, testCase.SectorMap.MF_wo_C19), ...
                'MF_wo_C19 should not include C19 (index 10)');
        end

        function testB_C19_IncludesCorrectSectors(testCase)
            % B_C19 should include B (index 4) and C19 (index 10)
            testCase.verifyEqual(sort(testCase.SectorMap.B_C19), [4, 10]);
            testCase.verifyEqual(testCase.NACE_55(4), "B");
            testCase.verifyEqual(testCase.NACE_55(10), "C19");
        end

        %% NACE Code Tests
        function testNACECodesStartCorrectly(testCase)
            % Verify first few NACE codes
            testCase.verifyEqual(testCase.NACE_55(1), "A01");
            testCase.verifyEqual(testCase.NACE_55(4), "B");
            testCase.verifyEqual(testCase.NACE_55(27), "F");
        end

        function testNACECodesEndCorrectly(testCase)
            % Verify last NACE codes
            testCase.verifyEqual(testCase.NACE_55(54), "R_S");
            testCase.verifyEqual(testCase.NACE_55(55), "T");
        end
    end
end
