function info = projectionsetup(info, variable, type, value)

% Sets default projection path.
%
% INPUTS
% - info       [cell]              1m cell of 12 char arrays (country codes) for initialization mode, or n4 cell array.
% - variable   [char,dseries]      1p char array, name of a variable, or dseries object (initialization mode).
% - type       [char]              1q char array, type of projection, possible values: 'Trend', 'Constant', and 'AR'.
% - value      [double]            scalar or row vector, value of the growth factor of the trend, or of the constant, or of the autoregressive parameters.
%
% OUTPUTS
% - info       [cell]      n4 array. First column contains names of the variables (exogenous or endogenous), second column specifies the type of projection
%                          ('Trend', 'Constant', or 'AR'), the third column stores parameters for the projection (growth factor, constant value, or
%                          autoregressive parameters), the fourth column describes the type of trend in a row character array ('Constant', 'Real', 'Nominal', ...)
%
% REMARKS
% This routine considers two modes:
%  - initialization mode. Takes two input arguments: a cell array with a list of country codes and a database containing all the observed exogenous
%                         variables and endogenous variables. The routine returns a cell array describing the projection path for all the variables.
%  - update mode.         Takes four input arguments: a cell array describing the projection path for all the variables (the output of the initialization
%                         mode), the name of the variable for which we want to change the projection path properties, the type of projection path
%                         (`Trend`, `Constant` and `AR`), the parameters of the projection path.
%
% EXAMPLE
% Assuming we have a database (dseries object) db containing all the required variables:
%
% >> info = projectionsetup({'U2'}, db);
%
% then we can project the variables by calling the dseries' projection routine:
%
% >> db.projection(info, 2000);
%
% where the last argument is the length of the projection.

% The following two boolean variables defines how the growth rates are computed.
exactgrowthrates = true;
addgrowthrates = false;

if nargin<=3
    %
    % Set defaults.
    %
    CC = info;            % Country codes in a cell array.
    db = variable;        % database in dseries object.
    calibmodfile = type;  % Full path the mod file where the calibrated parameters are defined.  
    % Load parameter values
    fid = fopen(calibmodfile, 'r');
    cline = fgetl(fid);
    calibration = false;
    while ischar(cline)
        if ~calibration && isequal(cline, '//+ CALIBRATION +')
            calibration = true;
            cline = fgetl(fid);
            continue
        end
        if calibration
            if isequal(cline, '//- CALIBRATION -')
                calibration = false;
                break
            end
            c = strfind(cline, '//');
            if ~isempty(c)
                % Remove comments
                cline(c:end) = '';
            end
            eval(cline);
        end
        cline = fgetl(fid);
    end
    fclose(fid);
    growth.tfp = u2_annual_tfp_growth_rate;
    growth.pri = u2_annual_inflation_rate;
    growth.pop = u2_annual_pop_growth_rate;
    % Annual Growth factors
    GROWTH.tfp = 1.0+growth.tfp/100;
    GROWTH.pri = 1.0+growth.pri/100;
    GROWTH.pop = 1.0+growth.pop/100;
    % Quaterly growth factors
    QGROWTH.tfp = nthroot(GROWTH.tfp, 4);
    QGROWTH.pri = nthroot(GROWTH.pri, 4);
    QGROWTH.pop = nthroot(GROWTH.pop, 4);
    % Quaterly growth rates
    if exactgrowthrates
        qgrowth.tfp = (QGROWTH.tfp-1)*100;
        qgrowth.pri = (QGROWTH.pri-1)*100;
        qgrowth.pop = (QGROWTH.pop-1)*100;
    else
        qgrowth.tfp = 100*log(QGROWTH.tfp);
        qgrowth.pri = 100*log(QGROWTH.pri);
        qgrowth.pop = 100*log(QGROWTH.pop);
    end
    % Nominal interest rate target
    NominalInterestRate = 3.2; % (to be updated if changed in the model)
    % Exponential trends
    Tfp = GROWTH.tfp;
    Population = GROWTH.pop;
    Real = Population*Tfp;
    Price = GROWTH.pri;
    Nominal = Real*Price;
    NominalPerCapita = Tfp*Price;
    QTfp = QGROWTH.tfp;
    QPopulation = QGROWTH.pop;
    QReal = QPopulation*QTfp;
    QPrice = QGROWTH.pri;
    QNominal = QReal*QPrice;
    QNominalPerCapita = QTfp*QPrice;
    %
    % Define quaterly growth rates
    %
    if exactgrowthrates
        if addgrowthrates
            qReal = qgrowth.tfp+qgrowth.pop;
            qNominal = qReal+qgrowth.pri;
            qNominalPerCapita = qgrowth.tfp+qgrowth.pri;
        else
            qReal = 100*(QReal-1);
            qNominal = 100*(QNominal-1);
            qNominalPerCapita = 100*(QNominalPerCapita-1);
        end
    else
        if addgrowthrates
            qReal = qgrowth.tfp+qgrowth.pop;
            qNominal = qReal+qgrowth.pri;
            qNominalPerCapita = qgrowth.tfp+qgrowth.pri;
        else
            qReal = 100*log(QReal);
            qNominal = 100*log(QNominal);
            qNominalPerCapita = 100*log(QNominalPerCapita);
        end
    end
    %
    % Define annual growth rates
    %
    if exactgrowthrates
        if addgrowthrates
            gReal = growth.tfp+growth.pop;
            gNominal = gReal+growth.pri;
            gNominalPerCapita = growth.tfp+growth.pri;
        else
            gReal = 100*(Real-1);
            gNominal = 100*(Nominal-1);
            gNominalPerCapita = 100*(NominalPerCapita-1);
        end
    else
        if addgrowthrates
            gReal = growth.tfp+growth.pop;
            gNominal = gReal+growth.pri;
            gNominalPerCapita = growth.tfp+growth.pri;
        else
            gReal = 100*log(Real);
            gNominal = 100*log(Nominal);
            gNominalPerCapita = 100*log(NominalPerCapita);
        end
    end
    lr_u2_tp_10y = u2_tp_10y_constant/(1-u2_tp_10y_L1);
    % load dataset.
    db = db(db.firstobservedperiod():db.lastobservedperiod());
    % Initialize the output
    info = cell(0,4);
    for i = 1:length(CC)
        cc = CC{i};
        info = vertcat(info, ...
                       {sprintf('%s_NFAN', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_IR_TAN', cc), 'Constant', NaN, 'Constant'}, ...
                       {sprintf('%s_IR_TLN', cc), 'Constant', NaN, 'Constant'}, ...
                       {sprintf('%s_PROPERTY_PAC_PE', cc), 'Constant', NaN, 'Constant'}, ...
                       {sprintf('%s_BU_Y_ITR', cc), 'Constant', NaN, 'Constant'}, ...
                       {sprintf('%s_BU_U_ITR', cc), 'Constant', NaN, 'Constant'}, ...
                       {sprintf('%s_BU_L_ITR', cc), 'Constant', NaN, 'Constant'}, ...
                       {sprintf('%s_BUSINESS_PAC_PE', cc), 'Constant', NaN, 'Constant'}, ...
                       {sprintf('%s_CONS_PAC_PE', cc), 'Constant', NaN, 'Constant'}, ...
                       {sprintf('%s_H_HH_Y_DIR', cc), 'Constant', NaN, 'Constant'}, ...
                       {sprintf('%s_H_HH_S_DIRL', cc), 'Constant', NaN, 'Constant'}, ...
                       {sprintf('%s_H_HH_S_DIRT', cc), 'Constant', NaN, 'Constant'}, ...
                       {sprintf('%s_RESIDENTIAL_PAC_PE', cc), 'Constant', NaN, 'Constant'}, ...
                       {sprintf('%s_RESIDENTIAL_PAC_AUX', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('%s_HH_U_IHR2', cc), 'Constant', NaN, 'Constant'}, ...
                       {sprintf('%s_HH_U_IHR', cc), 'Constant', NaN, 'Constant'}, ...
                       {sprintf('%s_HOUSEPRICE_PAC_PE', cc), 'Constant', NaN, 'Constant'}, ...
                       {sprintf('%s_Y_SIVR', cc), 'Constant', NaN, 'Constant'}, ...
                       {sprintf('%s_T_Y_SIVR', cc), 'Constant', NaN, 'Constant'}, ...
                       {sprintf('%s_LABOUR_PAC_PE', cc), 'Constant', NaN, 'Constant'}, ...
                       {sprintf('%s_C_LNH', cc), 'Constant', NaN, 'Constant'}, ...
                       {sprintf('%s_HH_SL_CON', cc), 'Constant', NaN, 'Constant'}, ...
                       {sprintf('%s_HH_SL_IHN', cc), 'Constant', NaN, 'Constant'}, ...
                       {sprintf('%s_SLRN', cc), 'Constant', NaN, 'Constant'}, ...
                       {sprintf('%s_SCB', cc), 'Constant', NaN, 'Constant'}, ...
                       {sprintf('%s_SCOE', cc), 'Constant', NaN, 'Constant'}, ...
                       {sprintf('%s_HH_SL_DEP', cc), 'Constant', NaN, 'Constant'}, ...
                       {sprintf('%s_TP_10Y', cc), 'Constant', lr_u2_tp_10y, 'Constant'}, ...
                       {sprintf('%s_TP_10YEXO', cc), 'Constant', 0, 'Constant'}, ...
                       {sprintf('%s_TP_10YENDO', cc), 'Constant', lr_u2_tp_10y, 'Constant'}, ...
                       {sprintf('%s_OIS_10Y', cc), 'Constant', NominalInterestRate+lr_u2_tp_10y, 'Constant'}, ...
                       {sprintf('%s_US_10Y', cc), 'Constant', NominalInterestRate+lr_u2_tp_10y, 'Constant'}, ...
                       {sprintf('%s_US_10Y_EXO', cc), 'Constant', NominalInterestRate+lr_u2_tp_10y, 'Constant'}, ...
                       {sprintf('%s_HH_L_CON', cc), 'Constant', NaN, 'Constant'}, ...
                       {sprintf('%s_HH_CRFL_CON', cc), 'Constant', NaN, 'Constant'}, ...
                       {sprintf('%s_HH_L_IHN', cc), 'Constant', NaN, 'Constant'}, ...
                       {sprintf('%s_HH_L_DEP', cc), 'Constant', NaN, 'Constant'}, ...
                       {sprintf('%s_LRN', cc), 'Constant', NaN, 'Constant'}, ...
                       {sprintf('%s_COE', cc), 'Constant', NaN, 'Constant'}, ...
                       {sprintf('%s_CB', cc), 'Constant', NaN, 'Constant'}, ...
                       {sprintf('%s_USTP_10Y', cc), 'Constant', NaN, 'Constant'}, ...
                       {sprintf('%s_BU_L_ITN', cc), 'Constant', NaN, 'Constant'},...
                       {sprintf('%s_R_LFP', cc), 'Constant', NaN, 'Constant'}, ...
                       {sprintf('%s_R_LUN', cc), 'Constant', NaN, 'Constant'}, ...
                       {sprintf('%s_T_BU_Y_Q_ITR', cc), 'Constant', NaN, 'Constant'}, ...
                       {sprintf('%s_T_BU_Y_Q_ITR_100', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('%s_BU_OITR_T', cc), 'Constant', NaN, 'Constant'}, ...
                       {sprintf('%s_H_HH_S_DIRP', cc), 'Constant', NaN, 'Constant'}, ...
                       {sprintf('%s_S_RPPD', cc), 'Constant', NaN, 'Constant'}, ...
                       {sprintf('%s_OS_RPPD', cc), 'Constant', NaN, 'Constant'}, ...
                       {sprintf('%s_S_RPPD_IHD', cc), 'Constant', NaN, 'Constant'},  ...
                       {sprintf('%s_HH_S_IHD', cc), 'Constant', NaN, 'Constant'}, ...
                       {sprintf('%s_S_GO_COR', cc), 'Constant', NaN, 'Constant'}, ...
                       {sprintf('%s_DL_HH_L_CON', cc), 'Constant', NaN, 'Constant'}, ...
                       {sprintf('%s_BU_ITR', cc), 'Trend', Real, 'Real'}, ...
                       {sprintf('%s_BUSINESS_AUX', cc), 'Constant', log(1+qReal/100), 'Real'}, ...
                       {sprintf('%s_GO_COR', cc), 'Trend', Real, 'Real'}, ...
                       {sprintf('%s_AUX_GO_COR', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('%s_GO_COR_EXO', cc), 'Trend', Real, 'Real'}, ...
                       {sprintf('%s_GO_ITR', cc), 'Trend', Real, 'Real'}, ...
                       {sprintf('%s_HH_COR', cc), 'Trend', Real, 'Real'}, ...
                       {sprintf('%s_HH_DIR', cc), 'Trend', Real, 'Real'}, ...
                       {sprintf('%s_HH_EDIRP', cc), 'Trend', Real, 'Real'}, ...
                       {sprintf('%s_HH_IHR', cc), 'Trend', Real, 'Real'}, ...
                       {sprintf('%s_ITR', cc), 'Trend', Real, 'Real'}, ...
                       {sprintf('%s_IVR', cc), 'Trend', Real, 'Real'}, ...
                       {sprintf('%s_MTR', cc), 'Trend', Real, 'Real'}, ...
                       {sprintf('%s_SIVR', cc), 'Trend', Real, 'Real'}, ...
                       {sprintf('%s_T_YER', cc), 'Trend', Real, 'Real'}, ...
                       {sprintf('%s_TBR', cc), 'Trend', Real, 'Real'}, ...
                       {sprintf('%s_WDREX', cc), 'Trend', Real, 'Real'}, ...
                       {sprintf('%s_AUX_WDREX', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('%s_WDREX_EXO', cc), 'Trend', Real, 'Real'}, ...
                       {sprintf('%s_WDR', cc), 'Trend', Real, 'Real'}, ...
                       {sprintf('%s_WDRIN', cc), 'Trend', Real, 'Real'}, ...
                       {sprintf('%s_XTR', cc), 'Trend', Real, 'Real'}, ...
                       {sprintf('%s_YER', cc), 'Trend', Real, 'Real'}, ...
                       {sprintf('%s_HH_TWR', cc), 'Trend', Real, 'Real'}, ...
                       {sprintf('%s_HH_DIRP', cc), 'Trend', Real, 'Real'}, ...
                       {sprintf('%s_HH_C_DDR', cc), 'Trend', Real, 'Real'}, ...
                       {sprintf('%s_SKHR', cc), 'Trend', Real, 'Real'}, ...
                       {sprintf('%s_T_GO_PUR', cc), 'Trend', Real, 'Real'}, ...
                       {sprintf('%s_GO_PUR', cc), 'Trend', Real, 'Real'}, ...
                       {sprintf('%s_T_GO_CER', cc), 'Trend', Real, 'Real'}, ...
                       {sprintf('%s_GO_CER', cc), 'Trend', Real, 'Real'}, ...
                       {sprintf('%s_T_GO_ITR', cc), 'Trend', Real, 'Real'}, ...
                       {sprintf('%s_GO_RCOR', cc), 'Trend', Real, 'Real'}, ...
                       {sprintf('%s_GO_ORCOR', cc), 'Trend', Real, 'Real'}, ...
                       {sprintf('%s_HH_OIHR', cc), 'Trend', Real, 'Real'}, ...
                       {sprintf('%s_SKFR', cc), 'Trend', Real, 'Real'}, ...
                       {sprintf('%s_HH_OCOR', cc), 'Trend', Real, 'Real'}, ...
                       {sprintf('%s_HH_EDIR', cc), 'Trend', Real, 'Real'}, ...
                       {sprintf('%s_HH_EDIRL', cc), 'Trend', Real, 'Real'}, ...
                       {sprintf('%s_HH_EDIRT', cc), 'Trend', Real, 'Real'}, ...
                       {sprintf('%s_HH_DIRL', cc), 'Trend', Real, 'Real'}, ...
                       {sprintf('%s_HH_DIRT', cc), 'Trend', Real, 'Real'}, ...
                       {sprintf('%s_BU_OITR', cc), 'Trend', Real, 'Real'}, ...
                       {sprintf('%s_MST', cc), 'Trend', Real, 'Real'}, ...
                       {sprintf('%s_XNR', cc), 'Trend', Real, 'Real'}, ...
                       {sprintf('%s_XXST', cc), 'Trend', Real, 'Real'}, ...
                       {sprintf('%s_XXR', cc), 'Trend', Real, 'Real'}, ...
                       {sprintf('%s_MXST', cc), 'Trend', Real, 'Real'}, ...
                       {sprintf('%s_MXR', cc), 'Trend', Real, 'Real'}, ...
                       {sprintf('%s_WER', cc), 'Trend', Real, 'Real'}, ...
                       {sprintf('%s_WEREX', cc), 'Trend', Real, 'Real'}, ...
                       {sprintf('%s_MNR', cc), 'Trend', Real, 'Real'}, ...
                       {sprintf('%s_HH_C_ODDR', cc), 'Trend', Real, 'Real'}, ...
                       {sprintf('%s_GO_RCON', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_GO_TEN', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_GO_REN', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_GO_IPN', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_GO_MAL', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_GO_B9N', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_GO_CEN', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_CEN', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_GO_CON', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_GO_HH_DTN', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_GO_ITN', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_GO_SBCN', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_GO_SCN', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_GO_TIN', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_GOSMIN', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_HH_B_SCN', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_HH_CON', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_HH_DIN', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_HH_DINP', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_HH_DTN', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_HH_IHN', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_IVN', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_MIN', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_TBN', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_YEN', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_HH_GOSMIN', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_HH_C_DDN', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_HH_GOS', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_HH_B_IRN', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_HH_B_OPPN', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_T_GO_RRN', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_GO_RRN', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_GO_BU_DTN', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_GO_RW_DTN', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_GO_DTN', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_GO_TRN', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_T_GO_SBCN', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_GO_PUN', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_T_GO_SIN', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_GO_SIN', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_T_YEN', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_HH_B_APN', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_HH_GSN', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_HH_B_KTN', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_HH_IVN', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_HH_ITN', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_HH_KFN', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_HH_NPN', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_HH_KFNPN', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_HH_CFKN', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_HH_B9N', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_HH_TWN', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_HWN', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_HH_FWN', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_W0_CEN', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_HH_B_SBN', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_HH_DINL', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_HH_DINT', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_WAN', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_XNN', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_XTN', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_MTN', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_XXN', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_MXN', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_MNN', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_BU_ITN', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_TIN', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_GO_BU_SCN', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_GO_RW_SCN', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_ITN', cc), 'Trend', Nominal, 'Nominal'}, ...
                       {sprintf('%s_GO_COD', cc), 'Trend', Price, 'Price'}, ...
                       {sprintf('%s_GO_CED', cc), 'Trend', Price, 'Price'}, ...
                       {sprintf('%s_GO_C_CEN', cc), 'Trend', Price, 'Price'}, ...
                       {sprintf('%s_BU_ITD', cc), 'Trend', Price, 'Price'}, ...
                       {sprintf('%s_CMEDEX', cc), 'Trend', Price, 'Price'}, ...
                       {sprintf('%s_CMEDEX_AUX', cc), 'Constant', 1.0, 'Constant'}, ...
                       {sprintf('%s_CMEDEX_EXO', cc), 'Trend', Price, 'Price'}, ...
                       {sprintf('%s_CXEDEX', cc), 'Trend', Price, 'Price'}, ...
                       {sprintf('%s_CXEDEX_AUX', cc), 'Constant', 1.0, 'Constant'}, ...
                       {sprintf('%s_CXEDEX_EXO', cc), 'Trend', Price, 'Price'}, ...
                       {sprintf('%s_GO_ITD', cc), 'Trend', Price, 'Price'}, ...
                       {sprintf('%s_HEF', cc), 'Trend', Price, 'Price'}, ...
                       {sprintf('%s_HEG', cc), 'Trend', Price, 'Price'}, ...
                       {sprintf('%s_HEX', cc), 'Trend', Price, 'Price'}, ...
                       {sprintf('%s_HICP', cc), 'Trend', Price, 'Price'}, ...
                       {sprintf('%s_HH_COD', cc), 'Trend', Price, 'Price'}, ...
                       {sprintf('%s_HH_IHD', cc), 'Trend', Price, 'Price'}, ...
                       {sprintf('%s_HIC', cc), 'Trend', Price, 'Price'}, ...
                       {sprintf('%s_HIF', cc), 'Trend', Price, 'Price'}, ...
                       {sprintf('%s_ITD', cc), 'Trend', Price, 'Price'}, ...
                       {sprintf('%s_MXD', cc), 'Trend', Price, 'Price'}, ...
                       {sprintf('%s_MXDNO', cc), 'Trend', Price, 'Price'}, ...
                       {sprintf('%s_PCU', cc), 'Trend', Price, 'Price'}, ...
                       {sprintf('%s_PCU_EXO', cc), 'Trend', Price, 'Price'}, ...
                       {sprintf('%s_PCU_AUX', cc), 'Constant', 1.0, 'Constant'}, ...
                       {sprintf('%s_POU', cc), 'Trend', Price, 'Price'}, ...
                       {sprintf('%s_POU_EXO', cc), 'Trend', Price, 'Price'}, ...
                       {sprintf('%s_POU_AUX', cc), 'Constant', 1.0, 'Constant'}, ...
                       {sprintf('%s_RPPD', cc), 'Trend', Price, 'Price'}, ...
                       {sprintf('%s_YED', cc), 'Trend', Price, 'Price'}, ...
                       {sprintf('%s_YWD', cc), 'Trend', Price, 'Price'}, ...
                       {sprintf('%s_OHEG', cc), 'Trend', Price, 'Price'}, ...
                       {sprintf('%s_OHEF', cc), 'Trend', Price, 'Price'}, ...
                       {sprintf('%s_OHEX', cc), 'Trend', Price, 'Price'}, ...
                       {sprintf('%s_GO_OITD', cc), 'Trend', Price, 'Price'}, ...
                       {sprintf('%s_HH_OCOD', cc), 'Trend', Price, 'Price'}, ...
                       {sprintf('%s_BU_OITD', cc), 'Trend', Price, 'Price'}, ...
                       {sprintf('%s_HH_OIHD', cc), 'Trend', Price, 'Price'}, ...
                       {sprintf('%s_XTD', cc), 'Trend', Price, 'Price'}, ...
                       {sprintf('%s_MTD', cc), 'Trend', Price, 'Price'}, ...
                       {sprintf('%s_PMNST', cc), 'Trend', Price, 'Price'}, ...
                       {sprintf('%s_PXXST', cc), 'Trend', Price, 'Price'}, ...
                       {sprintf('%s_XXD', cc), 'Trend', Price, 'Price'}, ...
                       {sprintf('%s_PMXST', cc), 'Trend', Price, 'Price'}, ...
                       {sprintf('%s_CXDEX', cc), 'Trend', Price, 'Price'}, ...
                       {sprintf('%s_CMDEX', cc), 'Trend', Price, 'Price'}, ...
                       {sprintf('%s_CXD', cc), 'Trend', Price, 'Price'}, ...
                       {sprintf('%s_CMD', cc), 'Trend', Price, 'Price'}, ...
                       {sprintf('%s_MED', cc), 'Trend', Price, 'Price'}, ...
                       {sprintf('%s_MEED', cc), 'Trend', Price, 'Price'}, ...
                       {sprintf('%s_XND', cc), 'Trend', Price, 'Price'}, ...
                       {sprintf('%s_MND', cc), 'Trend', Price, 'Price'}, ...
                       {sprintf('%s_TFPLG', cc), 'Trend', Tfp, 'Tfp'}, ...
                       {sprintf('%s_C_CER', cc), 'Trend', Tfp, 'Tfp'}, ...
                       {sprintf('%s_T_PRO', cc), 'Trend', Tfp, 'Tfp'}, ...
                       {sprintf('%s_C_CER_TAR', cc), 'Trend', Tfp, 'Tfp'}, ...
                       {sprintf('%s_C_WAR', cc), 'Trend', Tfp, 'Tfp'}, ...
                       {sprintf('%s_C_CEN', cc), 'Trend', NominalPerCapita, 'NominalPerCapita'}, ...
                       {sprintf('%s_C_GO_BU_SCN', cc), 'Trend', NominalPerCapita, 'NominalPerCapita'}, ...
                       {sprintf('%s_C_WAN', cc), 'Trend', NominalPerCapita, 'NominalPerCapita'}, ...
                       {sprintf('%s_LEN', cc), 'Trend', Population, 'Population'}, ...
                       {sprintf('%s_LNN', cc), 'Trend', Population, 'Population'}, ...
                       {sprintf('%s_GO_LNN', cc), 'Trend', Population, 'Population'}, ...
                       {sprintf('%s_T_LNN', cc), 'Trend', Population, 'Population'}, ...
                       {sprintf('%s_WAP', cc), 'Trend', Population, 'Population'}, ...
                       {sprintf('%s_WAP_ANNUAL_GROWTH', cc), 'Constant', u2_annual_pop_growth_rate/100, 'Constant'}, ...
                       {sprintf('%s_TAU', cc), 'Trend', Population, 'Population'}, ...
                       {sprintf('%s_Q_TAU', cc), 'Constant', qgrowth.pop/100, 'Constant'}, ...
                       {sprintf('%s_OLNN', cc), 'Trend', Population, 'Population'}, ...
                       {sprintf('%s_LSN', cc), 'Trend', Population, 'Population'}, ...
                       {sprintf('%s_OLSN', cc), 'Trend', Population, 'Population'}, ...
                       {sprintf('%s_LFN', cc) 'Trend', Population, 'Population'}, ...
                       {sprintf('%s_EENM', cc), 'Constant', 1.0, 'Constant'}, ...
                       {sprintf('%s_REENX', cc), 'Constant', 1.0, 'Constant'}, ...
                       {sprintf('%s_EXR', cc), 'Constant', 1.0, 'Constant'}, ...
                       {sprintf('%s_EENX', cc), 'Constant', 1.0, 'Constant'}, ...
                       {sprintf('%s_EENX_tmp', cc), 'Constant', NaN, 'Constant'}, ...
                       {sprintf('%s_Y_ITD', cc), 'Constant', 1.0, 'Constant'}, ...
                       {sprintf('%s_T_PRO_STATIONARY_SHOCK', cc), 'Constant', 1.0, 'Constant'}, ...
                       {'DYN_H_TRADE', 'Constant', 1.0, 'Constant'}, ...
                       {'DUMMY_CRISIS', 'Constant', 1.0, 'Constant'}, ...
                       {'dummy_long_run_EENX', 'Constant', 1.0, 'Constant'}, ...
                       {'dummy_long_run_EEN', 'Constant', 1.0, 'Constant'}, ...
                       {sprintf('%s_TTREND', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('%s_TREND_OHEX', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('%s_TREND_OHEG', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('%s_OS_RPPD_T', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('%s_G_A_YED', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('%s_G_C_CER', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('%s_T_G_C_CER', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('%s_G_HH_S_DIRP', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('%s_G_HH_Y_VDIR', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('%s_G_HH_Y_VDIR_VE', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('%s_G_R_LUN', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('%s_G_VYER', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('%s_G_VYER_VE', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('%s_G_YER', cc), 'Constant', 0.0, 'Constant'}, ...
                       {'dummy_long_run', 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_T_BU_Y_Q_ITR', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_HGPDR', cc), 'Constant', 0.0, 'Constant'}, ...
                       {'DYN_H_HICP', 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_T_R_LUN', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_T_C_LNH', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_T_R_LFP', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_LSN', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_C_LNH', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_TAU', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('%s_G_EYER', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('%s_G_HH_S_EDIRL', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('%s_G_HH_S_EDIRP', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('%s_G_HH_S_EDIRT', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('%s_G_HH_Y_EDIR', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('%s_G_LNN', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_H_HH_S_DIRP', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_T_Y_SIVR', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_NFAN', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_EENX', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_EEN_RES', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_HH_IVN', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_HH_ITN', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_HH_D_IHN', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_H_Q_YER', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_H_HH_Y_DIR', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_H_HH_S_DIRL', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_H_HH_S_DIRT', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_W0_CEN', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_MIN', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_HH_B_SCN', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_HH_DTN', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_HH_B_SBN', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_S_DTNL' , cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_S_DTNT', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_HH_L_CON', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_HH_L_IHN', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_HH_L_DEP', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_LRN', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_COE', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_CB', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_G_YER_E10Y', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_STN_E10Y', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_CMDEX', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_TIN', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_GO_CON', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_YER', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('%s_G_HH_S_VDIRP', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('%s_G_HH_S_VDIRP_VE', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('%s_ZHICP', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('%s_ZHIF', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('%s_IR_TAN_RV', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('%s_IR_TLN_RV', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('%s_NFANRV_S', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('%s_Y_DNFAN', cc), 'Constant', NaN, 'Constant'}, ...
                       {sprintf('%s_NFANRV', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('%s_Q16_S_RPPD', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('%s_Q_S_RPPD_IHD', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('%s_ES_RPPD_IHD', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_ESTN', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('%s_EPS_L_%s_T_YER', cc, cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('%s_G_HH_S_VDIRL', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('%s_G_HH_S_VDIRL_VE', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('%s_G_HH_S_VDIRT', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('%s_G_HH_S_VDIRT_VE', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('%s_DLYHLT', cc), 'Constant', log((1+u2_annual_tfp_growth_rate/100)*(1+u2_annual_pop_growth_rate/100))/4, 'Constant'}, ...
                       {sprintf('%s_G_HH_Y_DIR', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('%s_G_HH_S_DIRL', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('%s_G_HH_S_DIRT', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_HH_EOCOR', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('%s_HGPDR', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('%s_G_YER_E10Y', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('%s_G_YER_E10Y_VE', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('%s_dUSTP_10Y', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_EHIC', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('%s_EEN_RES_C', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('%s_EEN_RES', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_IR_TAN', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_IR_TLN', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_EXR', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_EENM', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_BU_ITR', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_LNN', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_R_LFP', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_HH_GOS', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_HH_C_DDR', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_HH_B_IRN', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_HH_B_OPPN', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_SIVR', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_IR_TAN_RV', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_IR_TLN_RV', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_NFANRV_S', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_S_RPPD', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_HH_IHR', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_HH_B_APN', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_HH_B_KTN', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_HH_NPN', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_HWN', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_HH_FWN', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_Q_T_PRO', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_HH_COR', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_HH_SL_CON', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_HH_SL_IHN', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_SLRN', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_SCB', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_SCOE', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_HH_SL_DEP', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_TP_10Y', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_TP_10YENDO', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_USTP_10Y', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_A_C_WAN', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_HH_Q_COD', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_BU_Q_ITD', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_HH_Q_IHD', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_MTR', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_XNR', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_XNN', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_XXR', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_XXD', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_MXR', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_MXD', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_MND', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_A_YED', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_STN', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_BU_ITD_tmp', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_CMEDEX', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_CXEDEX', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_EENX_tmp', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_EEN_RES_C', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_GOSMIN', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_GO_BU_DTN', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_GO_BU_R_DTN', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_GO_BU_R_SCN', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_GO_CER', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_GO_COR', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_GO_C_CEN', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_GO_HH_DTN', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_GO_HH_R_DTN', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_GO_HH_R_SCN', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_GO_IPN', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_GO_ITD', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_GO_ITR', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_GO_LNN', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_GO_MAL', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_GO_PUR', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_GO_RCOR', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_GO_REN', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_GO_RRN', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_GO_RW_DTN', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_GO_RW_R_SCN', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_GO_R_TIN', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_GO_SBCN', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_GO_SIN', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_HEF', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_HEG', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_HEX', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_HH_COD_tmp', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_HH_COR_tmp', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_HH_C_DDR_tmp', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_HH_IHD_tmp', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_HH_OIHR', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_PCU', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_POU', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_S_ORPPD', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_TFPLG', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_T_BU_Y_Q_ITR_100', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_T_GO_BU_R_DTN', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_T_GO_BU_R_SCN', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_T_GO_CER', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_T_GO_HH_R_DTN', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_T_GO_HH_R_SCN', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_T_GO_ITR', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_T_GO_PUR', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_T_GO_RRN', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_T_GO_RW_R_DTN', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_T_GO_RW_R_SCN', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_T_GO_R_TIN', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_T_GO_SBCN', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_T_GO_SIN', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_T_Q_YER_100', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_US_10Y', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_US_EHIC', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_WDREX', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_WDR', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_ZHICP', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_ZHIF', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_G_C_CER', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_GO_COR', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_T_PRO_STATIONARY_SHOCK', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_WAP_ANNUAL_GROWTH', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('%s_D_IHR', cc), 'Constant', db{sprintf('%s_D_IHR', cc)}.data(end), 'Constant'}, ...
                       {sprintf('%s_S_DTNL', cc), 'Constant', eval(sprintf('%s_s_dtnl_lr', lower(cc))), 'Constant'}, ...
                       {sprintf('%s_S_DTNT', cc), 'Constant', eval(sprintf('%s_s_dtnt_lr', lower(cc))), 'Constant'}, ...
                       {sprintf('%s_S_RW_NFAN', cc), 'Constant', db{sprintf('%s_S_RW_NFAN', cc)}.data(end), 'Constant'}, ...
                       {sprintf('%s_S_SCNL', cc), 'Constant', db{sprintf('%s_S_SCNL', cc)}.data(end), 'Constant'}, ...
                       {sprintf('%s_S_SCNT', cc), 'Constant', db{sprintf('%s_S_SCNT', cc)}.data(end), 'Constant'}, ...
                       {sprintf('%s_S_TIH', cc), 'Constant', db{sprintf('%s_S_TIH', cc)}.data(end), 'Constant'}, ...
                       {sprintf('%s_T_GO_BU_R_ODTN', cc), 'Constant', db{sprintf('%s_T_GO_BU_R_ODTN', cc)}.data(end), 'Constant'}, ...
                       {sprintf('%s_T_GO_HH_R_ODTN', cc), 'Constant', db{sprintf('%s_T_GO_HH_R_ODTN', cc)}.data(end), 'Constant'}, ...
                       {sprintf('%s_T_GO_R_OTIN', cc), 'Constant', db{sprintf('%s_T_GO_R_OTIN', cc)}.data(end), 'Constant'}, ...
                       {sprintf('%s_T_R_LFP', cc), 'Constant', eval(sprintf('%s_t_r_lfp_lr', lower(cc))), 'Constant'}, ...
                       {sprintf('%s_T_R_LUN', cc), 'Constant', eval(sprintf('%s_t_r_lun_lr', lower(cc))), 'Constant'}, ...
                       {sprintf('%s_T_C_LNH', cc), 'Constant', eval(sprintf('%s_t_c_lnh_lr', lower(cc))), 'Constant'}, ...
                       {sprintf('%s_BU_D_ITR', cc), 'Constant', db{sprintf('%s_BU_D_ITR', cc)}.data(end), 'Constant'}, ...
                       {sprintf('%s_BU_S_DTN', cc), 'Constant', db{sprintf('%s_BU_S_DTN', cc)}.data(end), 'Constant'}, ...
                       {sprintf('%s_D_ITR', cc), 'Constant', db{sprintf('%s_D_ITR', cc)}.data(end), 'Constant'}, ...
                       {sprintf('%s_GO_Y_OSBCN', cc), 'Constant', db{sprintf('%s_GO_Y_OSBCN', cc)}.data(end), 'Constant'}, ...
                       {sprintf('%s_GO_Y_OSIN', cc), 'Constant', db{sprintf('%s_GO_Y_OSIN', cc)}.data(end), 'Constant'}, ...
                       {sprintf('%s_T_GO_R_TIN', cc), 'Constant', db{sprintf('%s_T_GO_R_OTIN', cc)}.data(end), 'Constant'}, ...
                       {sprintf('%s_GO_R_TIN', cc), 'Constant', db{sprintf('%s_T_GO_R_OTIN', cc)}.data(end), 'Constant'}, ...
                       {sprintf('%s_T_GO_RW_R_DTN', cc), 'Constant', db{sprintf('%s_T_GO_RW_R_DTN', cc)}.data(end), 'Constant'},  ...
                       {sprintf('%s_HH_D_IHN', cc), 'Constant', eval(sprintf('%s_hh_d_ihn_lr', lower(cc))), 'Constant'}, ...
                       {sprintf('%s_GO_R_IPN', cc), 'Constant', 100*mean(db{sprintf('%s_AMIR', cc)}.data(end-11:end))/4, 'Constant'}, ...
                       {sprintf('%s_T_GO_HH_R_DTN', cc), 'Constant', db{sprintf('%s_T_GO_HH_R_ODTN', cc)}.data(end), 'Constant'}, ...
                       {sprintf('%s_GO_HH_R_DTN', cc), 'Constant', db{sprintf('%s_T_GO_HH_R_ODTN', cc)}.data(end), 'Constant'}, ...
                       {sprintf('%s_T_GO_BU_R_DTN', cc), 'Constant', db{sprintf('%s_T_GO_BU_R_ODTN', cc)}.data(end), 'Constant'}, ...
                       {sprintf('%s_GO_BU_R_DTN', cc), 'Constant', db{sprintf('%s_T_GO_BU_R_ODTN', cc)}.data(end), 'Constant'}, ...
                       {sprintf('%s_T_GO_RW_R_SCN', cc), 'Constant', db{sprintf('%s_T_GO_RW_R_OSCN', cc)}.data(end), 'Constant'}, ...
                       {sprintf('%s_GO_RW_R_SCN', cc), 'Constant', db{sprintf('%s_T_GO_RW_R_OSCN', cc)}.data(end), 'Constant'}, ...
                       {sprintf('%s_T_GO_HH_R_SCN', cc), 'Constant', db{sprintf('%s_T_GO_HH_R_OSCN', cc)}.data(end), 'Constant'}, ...
                       {sprintf('%s_GO_HH_R_SCN', cc), 'Constant', db{sprintf('%s_T_GO_HH_R_OSCN', cc)}.data(end), 'Constant'}, ...
                       {sprintf('%s_T_GO_BU_R_SCN', cc), 'Constant', db{sprintf('%s_T_GO_BU_R_OSCN', cc)}.data(end), 'Constant'}, ...
                       {sprintf('%s_GO_BU_R_SCN', cc), 'Constant', db{sprintf('%s_T_GO_BU_R_OSCN', cc)}.data(end), 'Constant'}, ...
                       {sprintf('%s_Q_T_PRO', cc), 'Constant', QTfp, 'Constant'}, ...
                       {sprintf('%s_STN_E10Y', cc), 'Constant', NominalInterestRate, 'Constant'}, ...
                       {sprintf('%s_STN_E10Y_VE', cc), 'Constant', 40*NominalInterestRate, 'Constant'}, ...
                       {sprintf('%s_A_C_CER', cc), 'Constant', growth.tfp, 'Constant'}, ...
                       {sprintf('%s_A_C_CER_TAR', cc), 'Constant', growth.tfp, 'Constant'}, ...
                       {sprintf('%s_E_G_A_YED_VE', cc), 'Constant', growth.pri, 'Constant'}, ...
                       {sprintf('%s_G_C_CER_VE', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('%s_A_YED_VE', cc), 'Constant', growth.pri, 'Constant'}, ...
                       {sprintf('%s_A_YED', cc), 'Constant', growth.pri, 'Constant'}, ...
                       {sprintf('%s_A_COD', cc), 'Constant', growth.pri, 'Constant'}, ...
                       {sprintf('%s_A_YED_TAR', cc), 'Constant', growth.pri, 'Constant'}, ...
                       {sprintf('%s_A4_HH_COD', cc),  'Constant', growth.pri, 'Constant'}, ...
                       {sprintf('%s_A4_YED', cc),  'Constant', growth.pri, 'Constant'}, ...
                       {sprintf('%s_E_A_YED_1', cc), 'Constant', growth.pri, 'Constant'}, ...
                       {sprintf('%s_EHIC', cc), 'Constant', growth.pri, 'Constant'}, ...
                       {sprintf('%s_US_EHIC', cc),  'Constant', growth.pri, 'Constant'}, ...
                       {sprintf('%s_US_EHIC_EXO', cc),  'Constant', growth.pri, 'Constant'}, ...
                       {sprintf('%s_ESTN', cc), 'Constant', NominalInterestRate, 'Constant'}, ...
                       {sprintf('%s_STN', cc), 'Constant', NominalInterestRate, 'Constant'}, ...
                       {sprintf('%s_STN_ZLB', 'U2'), 'Constant', NominalInterestRate, 'Constant'}, ...
                       {sprintf('%s_AMIR', cc), 'Constant', mean(db{sprintf('%s_AMIR', cc)}.data(end-11:end)), 'Constant'}, ...
                       {sprintf('%s_BU_Y_MIBR', cc), 'Constant', .1570052172136647, 'Constant'}, ...
                       {sprintf('%s_T_Q_YER', cc), 'Constant', log((1+u2_annual_tfp_growth_rate/100)*(1+u2_annual_pop_growth_rate/100))/4*100, 'Constant'}, ...
                       {sprintf('%s_T_Q_YER_100', cc), 'Constant', log((1+u2_annual_tfp_growth_rate/100)*(1+u2_annual_pop_growth_rate/100))/4, 'Constant'}, ...
                       {sprintf('%s_H_Q_YER', cc), 'Constant', log((1+u2_annual_tfp_growth_rate/100)*(1+u2_annual_pop_growth_rate/100))/4*100, 'Constant'}, ...
                       {sprintf('%s_H_Q_YER400', cc), 'Constant', log((1+u2_annual_tfp_growth_rate/100)*(1+u2_annual_pop_growth_rate/100))/4, 'Constant'}, ...
                       {sprintf('%s_GO_Y_OCEN', cc), 'Constant', db{sprintf('%s_GO_Y_OCEN', cc)}(lastobservedperiod(db{sprintf('%s_GO_Y_OCEN', cc)})).data, 'Constant'}, ...
                       {sprintf('%s_GO_Y_OITN', cc), 'Constant', db{sprintf('%s_GO_Y_OITN', cc)}(lastobservedperiod(db{sprintf('%s_GO_Y_OITN', cc)})).data, 'Constant'}, ...
                       {sprintf('%s_GO_Y_OPUN', cc), 'Constant', db{sprintf('%s_GO_Y_OPUN', cc)}(lastobservedperiod(db{sprintf('%s_GO_Y_OPUN', cc)})).data, 'Constant'}, ...
                       {sprintf('%s_GO_Y_ORCON', cc), 'Constant', db{sprintf('%s_GO_Y_ORCON', cc)}(lastobservedperiod(db{sprintf('%s_GO_Y_ORCON', cc)})).data, 'Constant'}, ...
                       {sprintf('%s_GO_Y_OREN', cc), 'Constant', db{sprintf('%s_GO_Y_OREN', cc)}(lastobservedperiod(db{sprintf('%s_GO_Y_OREN', cc)})).data, 'Constant'}, ...
                       {sprintf('%s_T_GO_BU_R_OSCN', cc), 'Constant', db{sprintf('%s_T_GO_BU_R_OSCN', cc)}(lastobservedperiod(db{sprintf('%s_T_GO_BU_R_OSCN', cc)})).data, 'Constant'}, ...
                       {sprintf('%s_T_GO_HH_R_OSCN', cc), 'Constant', db{sprintf('%s_T_GO_HH_R_OSCN', cc)}(lastobservedperiod(db{sprintf('%s_T_GO_HH_R_OSCN', cc)})).data, 'Constant'}, ...
                       {sprintf('%s_T_GO_RW_R_OSCN', cc), 'Constant', db{sprintf('%s_T_GO_RW_R_OSCN', cc)}(lastobservedperiod(db{sprintf('%s_T_GO_RW_R_OSCN', cc)})).data, 'Constant'}, ...
                       {sprintf('%s_A_C_CEN', cc), 'Constant', gNominalPerCapita, 'Constant'}, ...
                       {sprintf('%s_T_A_C_CEN', cc), 'Constant', log((1+u2_annual_tfp_growth_rate/100))*100 + u2_annual_inflation_rate, 'Constant'}, ...
                       {sprintf('%s_A_C_WAN', cc), 'Constant', gNominalPerCapita, 'Constant'}, ...
                       {sprintf('%s_A_C_WAN_VE', cc), 'Constant', gNominalPerCapita, 'Constant'}, ...
                       {sprintf('%s_E_A_C_CEN_1', cc), 'Constant', gNominalPerCapita, 'Constant'}, ...
                       {sprintf('%s_DUMMY_TREND', cc), 'Constant', 1.0, 'Constant'}, ...
                       {sprintf('%s_TREND_MTR_AUX', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('%s_TREND_MND_AUX', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('%s_TREND_XXR_AUX', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('%s_TREND_XXD_AUX', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_TREND_MTR_AUX', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_TREND_MND_AUX', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_TREND_XXR_AUX', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_TREND_XXD_AUX', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('%s_FC', cc), 'Constant', 0.0, 'Constant'}, ...
                       {sprintf('res_%s_FC', cc), 'Constant', 0.0, 'Constant'});
        if ~strcmp(cc, 'U2')
            info = vertcat(info, ...
                           {sprintf('%s_EXR', 'U2'), 'Constant', 1.0, 'Constant'}, ...
                           {sprintf('%s_MED', 'U2'), 'Trend', Price, 'Price'}, ...
                           {sprintf('%s_PCU', 'U2'), 'Trend', Price, 'Price'}, ...
                           {sprintf('%s_POU', 'U2'), 'Trend', Price, 'Price'}, ...
                           {sprintf('%s_EHIC', 'U2'), 'Constant', growth.pri, 'Constant'}, ...
                           {sprintf('%s_A_YED', 'U2'), 'Constant', growth.pri, 'Constant'}, ...
                           {sprintf('%s_ESTN', 'U2'), 'Constant', NominalInterestRate, 'Constant'}, ...
                           {sprintf('%s_G_EYER', 'U2'), 'Constant', 0.0, 'Constant'}, ...
                           {sprintf('%s_G_YER', 'U2'), 'Constant', 0.0, 'Constant'}, ...
                           {sprintf('%s_G_YER_E10Y', 'U2'), 'Constant', 0.0, 'Constant'}, ...
                           {sprintf('%s_STN_E10Y', 'U2'), 'Constant', NominalInterestRate, 'Constant'}, ...
                           {sprintf('%s_dUSTP_10Y', 'U2'), 'Constant', 0.0, 'Constant'}, ...
                           {sprintf('%s_TP_10Y', 'U2'), 'Constant', lr_u2_tp_10y, 'Constant'}, ...
                           {sprintf('%s_OIS_10Y', 'U2'), 'Constant', NominalInterestRate+lr_u2_tp_10y, 'Constant'}, ...
                           {sprintf('%s_T_G_C_CER', 'U2'), 'Constant', 0, 'Constant'}, ...
                           {sprintf('%s_STN', 'U2'), 'Constant', NominalInterestRate, 'Constant'}, ...
                           {sprintf('%s_STN_ZLB', 'U2'), 'Constant', NominalInterestRate, 'Constant'}, ...
                           {sprintf('%s_DUMMY_TREND', 'U2'), 'Constant', 1.0, 'Constant'});
        end
        if strcmp(cc, 'NL')
            info = vertcat(info, ...
                           {sprintf('%s_ITD_OUTLIERS', cc), 'Constant', 0.0, 'Constant'});
        end
    end
    [~, idx] = unique(info(:,1));
    info = info(idx, :);
else
    %
    % Change defaults.
    %
    id = strcmp(variable, info(:,1));
    if isempty(variable)
        error('projection::set: Variable %s is unknown.', variable)
    else
        if ismember(type, {'Trend', 'Constant', 'AR'})
            info{id, 2} = type;
        else
            error('projection::set: Unknown projection method (%s).', type)
        end
        switch type
          case 'Trend'
            if isscalar(value) && isreal(value) && value>=0
                info{id, 3} = value;
            else
                error('projection::set: Growwth factor must be a real positive scalar.')
            end
          case 'Constant'
            if isscalar(value) && isreal(value)
                info{id, 3} = value;
            else
                error('projection::set: Constant must be a real scalar.')
            end
          case 'AR'
            if isrow(value) && isreal(value)
                info{id, 3} = value;
            else
                error('projection::set: Autoregressive parameters must be provided in a row vector (if p>1) or a scalar (p=1).')
            end
        end
    end
end