function [ts, TS] = computeirf(shockname, shockvalue, endogenousvariableslist, periods, baseline)

% INPUTS
% - shockname                [char]      1×n array, name of the shock.
% - shockvalue               [double]    scalar, value of shockname in period 1.
% - endogenousvariableslist  [cell]      1×m cell array of row char arrays, names of the endogenous variables.
% - periods                  [integer]   scalar, number of periods for the IRF
% - baseline                 [dseries]   Baseline around which the IRFs are computed.
%
% OUTPUT
% - ts                       [dseries]   object with m endogenous variables and periods+1 observations.
% - TS                       [dseries]   same as ts, but with a version of the model where the foreign variables are endogenous.

global M_ options_

if nargin<5 || isempty(baseline)
    try
        ds = dseries('../database/baseline.mat');
    catch
        if ~exist('../database/baseline.mat', 'file')
            computebaseline();
            ds = dseries('../database/baseline.mat');
        else
            if ~exist('dseries', 'class')
                error('Dynare is probably not in the path or you did not run dynare_config().')
            else
                error('Baseline file (../database/baseline.mat) probably corrupted.')
            end
        end
    end
end

if ~exist('+backward/driver.m', 'file')
    % Write the backward version of the model
    dynare('hybrid', '-DBACKWARD=true', '-DEXOGENIZE_FOREIGN=true', 'savemacro=backward.mod', 'onlymacro', 'notime');

    % Preprocess backward model
    dynare('backward', 'stochastic', 'fast', 'nopreprocessoroutput', 'notime');
end

try
    % Add the auxiliary variables to the baseline
    ds = backward.dynamic_set_auxiliary_series(ds, M_.params);
catch
    if ~exist('+backward/driver.m', 'file')
        % Write the backward version of the model
        dynare('hybrid', '-DBACKWARD=true', '-DEXOGENIZE_FOREIGN=true', 'savemacro=backward.mod', 'onlymacro', 'notime');
        % Preprocess backward model
        dynare('backward', 'stochastic', 'fast', 'nopreprocessoroutput', 'notime');
    else
        error('Something is wrong. There is probably a stalled version of backward.mod and +backward around.');
    end
end

% Build the path for the exogenous variable.
sdata = zeros(periods, 1);
sdata(1) = shockvalue;
shock = {dseries(sdata, ds.firstobservedperiod+M_.orig_maximum_lag+1, {shockname})}; % -> listofshocks


% Set options for nonlinear solver (use the fastest option).
options_.solve_algo = 14;
options_.maxiter = 500;
options_.tolx = 1e-8;
options_.tolf = 1e-8;

% Compute IRFs
[~, ~, irfs] = backward_model_irf(ds(ds.firstobservedperiod:ds.firstobservedperiod+M_.orig_maximum_lag), ds(ds.firstobservedperiod+M_.orig_maximum_lag+(1:periods)), shock, {}, periods);
ts = irfs.experiment_1{endogenousvariableslist{:}}(irfs.experiment_1.dates(M_.orig_maximum_lag+1:end));

% This is necessary, since for a still unknown reason the models are mixed on subsequent calls if not erased.
% Removing this file and subfolders forces dynare to recompile the model.
delete backward.mod
rmdir backward s
rmdir +backward s

if nargout>1

    %
    % Version of the model where the foreign variables are endogenous.
    %

    ds = dseries('../database/baseline.mat');

    if ~exist('+backward_endo_foreign/driver.m', 'file')
        % Write the backward version of the model
        dynare('hybrid', '-DBACKWARD=true', '-DEXOGENIZE_FOREIGN=false', 'savemacro=backward_endo_foreign.mod', 'onlymacro', 'notime');

        % Preprocess backward model
        dynare('backward_endo_foreign', 'stochastic', 'fast', 'nopreprocessoroutput', 'notime');
    end

    try
        % Add the auxiliary variables to the baseline
        ds = backward_endo_foreign.dynamic_set_auxiliary_series(ds, M_.params);
    catch
        if ~exist('+backward_endo_foreign/driver.m', 'file')
            % Write the backward version of the model
            dynare('hybrid', '-DBACKWARD=true', '-DEXOGENIZE_FOREIGN=false', 'savemacro=backward_endo_foreign.mod', 'onlymacro', 'notime');
            % Preprocess backward model
            dynare('backward_endo_foreign', 'stochastic', 'fast', 'nopreprocessoroutput', 'notime');
        else
            error('Something is wrong. There is probably a stalled version of backward_endo_foreign.mod and +backward_endo_foreign around.');
        end
    end

    % Build the path for the exogenous variable.
    sdata = zeros(periods, 1);
    sdata(1) = shockvalue;
    shock = {dseries(sdata, ds.firstobservedperiod+M_.orig_maximum_lag+1, {shockname})}; % -> listofshocks


    % Set options for nonlinear solver (use the fastest option).
    options_.solve_algo = 14;
    options_.maxiter = 500;
    options_.tolx = 1e-8;
    options_.tolf = 1e-8;

    % Compute IRFs
    [~, ~, irfs] = backward_model_irf(ds(ds.firstobservedperiod:ds.firstobservedperiod+M_.orig_maximum_lag), ds(ds.firstobservedperiod+M_.orig_maximum_lag+(1:periods)), shock, {}, periods);
    TS = irfs.experiment_1{endogenousvariableslist{:}}(irfs.experiment_1.dates(M_.orig_maximum_lag+1:end));

    % This is necessary, since for a still unknown reason the models are mixed on subsequent calls if not erased.
    % Removing this file and subfolders forces dynare to recompile the model.
    delete backward_endo_foreign.mod
    rmdir backward_endo_foreign s
    rmdir +backward_endo_foreign s

end