function [ts, TS, US, VS, WS] = computeirf(shockname, shockvalue, endogenousvariableslist, periods, baseline)

% INPUTS
% - shockname                [char]      1×n array, name of the shock.
% - shockvalue               [double]    scalar, value of shockname in period 1.
% - endogenousvariableslist  [cell]      1×m cell array of row char arrays, names of the endogenous variables.
% - periods                  [integer]   scalar, number of periods for the IRF
% - baseline                 [dseries]   Baseline around which the IRFs are computed.
%
% OUTPUT
% - ts                       [dseries]   object with m endogenous variables and periods+1 observations.
% - TS                       [dseries]   same as ts, but with a version of the model where the foreign variables are endogenous.
% - US                       [dseries]   same as ts, but with a version of the model where the exchange rate is exegonous.
% - VS                       [dseries]   same as ts, but with a version of the model where output gap has no effects on spreads.

global M_ options_

if nargin<5 || isempty(baseline)
    try
        ds = dseries('../database/baseline.mat');
    catch
        if ~exist('../database/baseline.mat', 'file')
            computebaseline();
            ds = dseries('../database/baseline.mat');
        else
            if ~exist('dseries', 'class')
                error('Dynare is probably not in the path or you did not run dynare_config().')
            else
                error('Baseline file (../database/baseline.mat) probably corrupted.')
            end
        end
    end
end

if ~exist('+backward/driver.m', 'file')
    % Write the backward version of the model
    dynare('hybrid', '-DBACKWARD=true', '-DEXOGENIZE_FOREIGN=true', '-DOUTPUT_GAP_AS_DEVIATION_TO_BASELINE_IN_SPREADS=true', 'savemacro=backward.mod', 'onlymacro', 'notime');

    % Preprocess backward model
    dynare('backward', 'stochastic', 'fast', 'nopreprocessoroutput', 'notime');
end

try
    % Add the auxiliary variables to the baseline
    ds = backward.dynamic_set_auxiliary_series(ds, M_.params);
catch
    if ~exist('+backward/driver.m', 'file')
        % Write the backward version of the model
        dynare('hybrid', '-DBACKWARD=true', '-DEXOGENIZE_FOREIGN=true', '-DOUTPUT_GAP_AS_DEVIATION_TO_BASELINE_IN_SPREADS=true', 'savemacro=backward.mod', 'onlymacro', 'notime');
        % Preprocess backward model
        dynare('backward', 'stochastic', 'fast', 'nopreprocessoroutput', 'notime');
    else
        error('Something is wrong. There is probably a stalled version of backward.mod and +backward around.');
    end
end

% Build the path for the exogenous variable.
sdata = zeros(periods, 1);
sdata(1) = shockvalue;
shock = {dseries(sdata, ds.firstobservedperiod+M_.orig_maximum_lag+1, {shockname})}; % -> listofshocks


% Set options for nonlinear solver (use the fastest option).
options_.solve_algo = 14;
options_.maxiter = 500;
options_.tolx = 1e-8;
options_.tolf = 1e-8;

% Compute IRFs
[~, ~, irfs] = backward_model_irf(ds(ds.firstobservedperiod:ds.firstobservedperiod+M_.orig_maximum_lag), ds(ds.firstobservedperiod+M_.orig_maximum_lag+(1:periods)), shock, {}, periods);
ts = irfs.experiment_1{endogenousvariableslist{:}}(irfs.experiment_1.dates(M_.orig_maximum_lag+1:end));

% This is necessary, since for a still unknown reason the models are mixed on subsequent calls if not erased.
% Removing this file and subfolders forces dynare to recompile the model.
delete backward.mod
rmdir backward s
rmdir +backward s

if nargout>1

    %
    % Version of the model where the foreign variables are endogenous.
    %

    ds = dseries('../database/baseline.mat');

    if ~exist('+backward_endo_foreign/driver.m', 'file')
        % Write the backward version of the model
        dynare('hybrid', '-DBACKWARD=true', '-DEXOGENIZE_FOREIGN=false', '-DOUTPUT_GAP_AS_DEVIATION_TO_BASELINE_IN_SPREADS=true', 'savemacro=backward_endo_foreign.mod', 'onlymacro', 'notime');

        % Preprocess backward model
        dynare('backward_endo_foreign', 'stochastic', 'fast', 'nopreprocessoroutput', 'notime');
    end

    try
        % Add the auxiliary variables to the baseline
        ds = backward_endo_foreign.dynamic_set_auxiliary_series(ds, M_.params);
    catch
        if ~exist('+backward_endo_foreign/driver.m', 'file')
            % Write the backward version of the model
            dynare('hybrid', '-DBACKWARD=true', '-DEXOGENIZE_FOREIGN=false', '-DOUTPUT_GAP_AS_DEVIATION_TO_BASELINE_IN_SPREADS=true', 'savemacro=backward_endo_foreign.mod', 'onlymacro', 'notime');
            % Preprocess backward model
            dynare('backward_endo_foreign', 'stochastic', 'fast', 'nopreprocessoroutput', 'notime');
        else
            error('Something is wrong. There is probably a stalled version of backward_endo_foreign.mod and +backward_endo_foreign around.');
        end
    end

    % Build the path for the exogenous variable.
    sdata = zeros(periods, 1);
    sdata(1) = shockvalue;
    shock = {dseries(sdata, ds.firstobservedperiod+M_.orig_maximum_lag+1, {shockname})}; % -> listofshocks


    % Set options for nonlinear solver (use the fastest option).
    options_.solve_algo = 14;
    options_.maxiter = 500;
    options_.tolx = 1e-8;
    options_.tolf = 1e-8;

    % Compute IRFs
    [~, ~, irfs] = backward_model_irf(ds(ds.firstobservedperiod:ds.firstobservedperiod+M_.orig_maximum_lag), ds(ds.firstobservedperiod+M_.orig_maximum_lag+(1:periods)), shock, {}, periods);
    TS = irfs.experiment_1{endogenousvariableslist{:}}(irfs.experiment_1.dates(M_.orig_maximum_lag+1:end));

    % This is necessary, since for a still unknown reason the models are mixed on subsequent calls if not erased.
    % Removing this file and subfolders forces dynare to recompile the model.
    delete backward_endo_foreign.mod
    rmdir backward_endo_foreign s
    rmdir +backward_endo_foreign s

    if nargout>2

        %
        % Version of the model where the exchange rate is exogenous
        %

        ds = dseries('../database/baseline.mat');

        if ~exist('+backward_exogenous_exchange_rate/driver.m', 'file')
            % Write the backward version of the model
            dynare('hybrid', '-DBACKWARD=true', '-DEXOGENIZE_FOREIGN=true', '-DOUTPUT_GAP_AS_DEVIATION_TO_BASELINE_IN_SPREADS=true', '-DEXOGENOUS_EXCHANGE_RATE=true', 'savemacro=backward_exogenous_exchange_rate.mod', 'onlymacro', 'notime');

            % Preprocess backward model
            dynare('backward_exogenous_exchange_rate', 'stochastic', 'fast', 'nopreprocessoroutput', 'notime');
        end

        try
            % Add the auxiliary variables to the baseline
            ds = backward_exogenous_exchange_rate.dynamic_set_auxiliary_series(ds, M_.params);
        catch
            if ~exist('+backward_exogenous_exchange_rate/driver.m', 'file')
                % Write the backward version of the model
                dynare('hybrid', '-DBACKWARD=true', '-DEXOGENIZE_FOREIGN=true', '-DOUTPUT_GAP_AS_DEVIATION_TO_BASELINE_IN_SPREADS=true', '-DEXOGENOUS_EXCHANGE_RATE=true', 'savemacro=backward_exogenous_exchange_rate.mod', 'onlymacro', 'notime');
                % Preprocess backward model
                dynare('backward_exogenous_exchange_rate', 'stochastic', 'fast', 'nopreprocessoroutput', 'notime');
            else
                error('Something is wrong. There is probably a stalled version of backward_exogenous_exchange_rate.mod and +backward_exogenous_exchange_rate around.');
            end
        end

        % Build the path for the exogenous variable.
        sdata = zeros(periods, 1);
        sdata(1) = shockvalue;
        shock = {dseries(sdata, ds.firstobservedperiod+M_.orig_maximum_lag+1, {shockname})}; % -> listofshocks


        % Set options for nonlinear solver (use the fastest option).
        options_.solve_algo = 14;
        options_.maxiter = 500;
        options_.tolx = 1e-8;
        options_.tolf = 1e-8;

        % Compute IRFs
        [~, ~, irfs] = backward_model_irf(ds(ds.firstobservedperiod:ds.firstobservedperiod+M_.orig_maximum_lag), ds(ds.firstobservedperiod+M_.orig_maximum_lag+(1:periods)), shock, {}, periods);
        US = irfs.experiment_1{endogenousvariableslist{:}}(irfs.experiment_1.dates(M_.orig_maximum_lag+1:end));

        % This is necessary, since for a still unknown reason the models are mixed on subsequent calls if not erased.
        % Removing this file and subfolders forces dynare to recompile the model.
        delete backward_exogenous_exchange_rate.mod
        rmdir backward_exogenous_exchange_rate s
        rmdir +backward_exogenous_exchange_rate s

        if nargout>3

            %
            % Version of the model where the exchange rate is exogenous
            %

            ds = dseries('../database/baseline.mat');

            if ~exist('+backward_no_output_gap_in_spreads/driver.m', 'file')
                % Write the backward version of the model
                dynare('hybrid', '-DBACKWARD=true', '-DEXOGENIZE_FOREIGN=true', '-DOUTPUT_GAP_AS_DEVIATION_TO_BASELINE_IN_SPREADS=true', '-DNO_OUTPUT_GAP_EFFECTS_ON_SPREADS=true', 'savemacro=backward_no_output_gap_in_spreads.mod', 'onlymacro', 'notime');

                % Preprocess backward model
                dynare('backward_no_output_gap_in_spreads', 'stochastic', 'fast', 'nopreprocessoroutput', 'notime');
            end

            try
                % Add the auxiliary variables to the baseline
                ds = backward_no_output_gap_in_spreads.dynamic_set_auxiliary_series(ds, M_.params);
            catch
                if ~exist('+backward_no_output_gap_in_spreads/driver.m', 'file')
                    % Write the backward version of the model
                    dynare('hybrid', '-DBACKWARD=true', '-DEXOGENIZE_FOREIGN=true', '-DOUTPUT_GAP_AS_DEVIATION_TO_BASELINE_IN_SPREADS=true', '-DNO_OUTPUT_GAP_EFFECTS_ON_SPREADS=true', 'savemacro=backward_no_output_gap_in_spreads.mod', 'onlymacro', 'notime');
                    % Preprocess backward model
                    dynare('backward_no_output_gap_in_spreads', 'stochastic', 'fast', 'nopreprocessoroutput', 'notime');
                else
                    error('Something is wrong. There is probably a stalled version of backward_no_output_gap_in_spreads.mod and +backward_no_output_gap_in_spreads around.');
                end
            end

            % Build the path for the exogenous variable.
            sdata = zeros(periods, 1);
            sdata(1) = shockvalue;
            shock = {dseries(sdata, ds.firstobservedperiod+M_.orig_maximum_lag+1, {shockname})}; % -> listofshocks


            % Set options for nonlinear solver (use the fastest option).
            options_.solve_algo = 14;
            options_.maxiter = 500;
            options_.tolx = 1e-8;
            options_.tolf = 1e-8;

            % Compute IRFs
            [~, ~, irfs] = backward_model_irf(ds(ds.firstobservedperiod:ds.firstobservedperiod+M_.orig_maximum_lag), ds(ds.firstobservedperiod+M_.orig_maximum_lag+(1:periods)), shock, {}, periods);
            VS = irfs.experiment_1{endogenousvariableslist{:}}(irfs.experiment_1.dates(M_.orig_maximum_lag+1:end));

            % This is necessary, since for a still unknown reason the models are mixed on subsequent calls if not erased.
            % Removing this file and subfolders forces dynare to recompile the model.
            delete backward_no_output_gap_in_spreads.mod
            rmdir backward_no_output_gap_in_spreads s
            rmdir +backward_no_output_gap_in_spreads s

            if nargout>4

                %
                % Version of the model where the exchange rate is exogenous
                %

                ds = dseries('../database/baseline.mat');

                if ~exist('+backward_no_output_gap_in_spreads_and_exogenous_exchange_rate/driver.m', 'file')
                    % Write the backward version of the model
                    dynare('hybrid', '-DBACKWARD=true', '-DEXOGENIZE_FOREIGN=true', '-DOUTPUT_GAP_AS_DEVIATION_TO_BASELINE_IN_SPREADS=true', '-DNO_OUTPUT_GAP_EFFECTS_ON_SPREADS=true', '-DEXOGENOUS_EXCHANGE_RATE=true', 'savemacro=backward_no_output_gap_in_spreads_x.mod', 'onlymacro', 'notime');

                    % Preprocess backward model
                    dynare('backward_no_output_gap_in_spreads_x', 'stochastic', 'fast', 'nopreprocessoroutput', 'notime');
                end

                try
                    % Add the auxiliary variables to the baseline
                    ds = backward_no_output_gap_in_spreads_x.dynamic_set_auxiliary_series(ds, M_.params);
                catch
                    if ~exist('+backward_no_output_gap_in_spreads_x/driver.m', 'file')
                        % Write the backward version of the model
                        dynare('hybrid', '-DBACKWARD=true', '-DEXOGENIZE_FOREIGN=true', '-DOUTPUT_GAP_AS_DEVIATION_TO_BASELINE_IN_SPREADS=true', '-DNO_OUTPUT_GAP_EFFECTS_ON_SPREADS=true', '-DEXOGENOUS_EXCHANGE_RATE=true', 'savemacro=backward_no_output_gap_in_spreads_x.mod', 'onlymacro', 'notime');
                        % Preprocess backward model
                        dynare('backward_no_output_gap_in_spreads_x', 'stochastic', 'fast', 'nopreprocessoroutput', 'notime');
                    else
                        error('Something is wrong. There is probably a stalled version of backward_no_output_gap_in_spreads_x.mod and +backward_no_output_gap_in_spreads_x around.');
                    end
                end

                % Build the path for the exogenous variable.
                sdata = zeros(periods, 1);
                sdata(1) = shockvalue;
                shock = {dseries(sdata, ds.firstobservedperiod+M_.orig_maximum_lag+1, {shockname})}; % -> listofshocks


                % Set options for nonlinear solver (use the fastest option).
                options_.solve_algo = 14;
                options_.maxiter = 500;
                options_.tolx = 1e-8;
                options_.tolf = 1e-8;

                % Compute IRFs
                [~, ~, irfs] = backward_model_irf(ds(ds.firstobservedperiod:ds.firstobservedperiod+M_.orig_maximum_lag), ds(ds.firstobservedperiod+M_.orig_maximum_lag+(1:periods)), shock, {}, periods);
                WS = irfs.experiment_1{endogenousvariableslist{:}}(irfs.experiment_1.dates(M_.orig_maximum_lag+1:end));

                % This is necessary, since for a still unknown reason the models are mixed on subsequent calls if not erased.
                % Removing this file and subfolders forces dynare to recompile the model.
                delete backward_no_output_gap_in_spreads_x.mod
                rmdir backward_no_output_gap_in_spreads_x s
                rmdir +backward_no_output_gap_in_spreads_x s

            end

        end

    end

end