function makeplots(filebasename, shockcode, shockname, variablecodes, variablenames, scenariocodes, scenarinames, scenariocolours, withbackward, compile2pdf, subplotsperrow)

% Build TikZ plots for ECB-BASE IRFs
%
% INPUTS
% - filebasename         [char]        1×s array, name of the file where the plots are saved.
% - shockcode            [char]        1×n array, code name of the shock (as defined in the main mod file).
% - shockname            [char]        1×m array, name of the shock (as displayed in the produced figure).
% - variablecodes        [cell]        1×p array of row char arrays, code name of the endogenous variables to be plotted (as defined in the main mod file).
% - variablenames        [cell]        1×p array of row char arrays, name of the endogenous variables (as displayed in the produced figure).
% - scenariocodes        [cell]        1×q array of row char arrays, code name of the scenario to be plotted (as defined in the main mod file).
% - scenarionames        [cell]        1×q array of row char arrays, name of the scenario to be plotted (as displayed in the produced figure).
% - scenariocolours      [cell]        1×q array of row char arrays, colour of the scenario to be plotted (as displayed in the produced figure).
% - withbackward         [logical]     Adds the simulation of the backward version of ECB-BASE if true (default value is true).
% - compile2pdf          [logical]     Compiles the generated TeX file if true (default value is true)
% - subplotperrow        [integer]     scalar, the number of sub figures per row (default is the integer part of the square root of 
%
% OUTPUTS
% None
%
% EXAMPLE
% In the models subfolder:
%
% >> makeplots('testfile', 'res_U2_STN', 'Shock on the nominal interest rate', ...
%          {'U2_STN', 'U2_A_YED', 'U2_BU_ITR', 'U2_EENX', 'U2_G_YER', 'U2_HH_COR', 'U2_LNN', 'U2_OIS_10Y', 'U2_SLRN', 'U2_TP_10Y', 'U2_YER'}, ...
%          {'Nominal interest rate', 'Annual inflation', 'Business investment', 'Nominal exchange rate', 'Output gap', ...
%           'Real consumption', 'Total employment', 'Euro Overnight Index Swap (10-year)', 'Spread on business lending rate', ...
%           'Ten years term premium', 'Real GDP'})

% stepan@adjemian.eu 27-02-2021

if nargin<10 || isempty(compile2pdf)
    compile2pdf = true;
end

if nargin<9
    compile2pdf = true;
    withbackward = true;
end

if nargin==9 && isempty(withbackward)
    withbackward = true;
end

if nargin<6
    scenariocodes = {'hybrid/full'};
    scenarionames = {'Full MCE'};
    scenariocolours = {'red'};
end

if withbackward
    scenariocodes{end+1} = 'backward';
    scenarionames{end+1} = 'Backward';
    scenariocolours{end+1} = 'black';
end

p = length(variablecodes); % number of subplots (# of endogenous variables)

if length(variablenames)~=p
    error('Second and third input arguments must be cell arrays with the same number of elements (names of the endogenous variables')
end

q = length(scenariocodes); % number of scenarii (# of curves in each subplot)
if nargin<11 || isempty(subplotsperrow)
    subplotsperrow = floor(sqrt(p));
end

scalingparam = 1/subplotsperrow;
numberofrows = ceil(q/subplotsperrow);

ROOT_FOLDER = getrootpath();

fid = fopen(sprintf('%s.tex', filebasename), 'w');
if fid ~= -1
    fprintf(fid, '\\documentclass[tikz]{standalone}\n');
    fprintf(fid, '\\usepackage{tikz,pgfplots}\n');
    fprintf(fid, '\\pgfplotsset{compat=newest}\n');
    fprintf(fid, '\\usetikzlibrary{patterns, matrix, arrows, decorations.pathreplacing, decorations.markings, calc}\n');
    fprintf(fid, '\\pgfplotsset{plot coordinates/math parser=false}\n');
    fprintf(fid, '\\usepgfplotslibrary{patchplots}\n');
    fprintf(fid, '\\newlength\\figureheight\n');
    fprintf(fid, '\\newlength\\figurewidth\n');
    fprintf(fid, '\\pgfplotsset{\n');
    fprintf(fid, 'scaled y ticks = false,\n');
    fprintf(fid, 'yticklabel style={/pgf/number format/fixed, /pgf/number format/precision=4},\n');
    fprintf(fid, 'table/search path={./irfs-data},\n');
    fprintf(fid, 'every tick label/.append style={font=\\tiny}\n');
    fprintf(fid, '}\n');
    fprintf(fid, '\n');
    fprintf(fid, '\\begin{document}\n');
    fprintf(fid, '\\begin{tikzpicture}\n');
    fprintf(fid, '\\matrix[ampersand replacement=\\&]{\n');
    irow = 1;
    for i=1:p
       fprintf(fid, '\\begin{axis}[scale=%f, title= \\tiny{%s}, enlargelimits=false, color=blue!30!black, thick, grid=major, grid style={line width=.2pt, draw=black!10}]\n', scalingparam, variablenames{i});
        for j=1:q
            fprintf(fid, '\\addplot[style={%s,mark=none}] table[x index=0, y index=1, col sep=space]{%sirfs-data/%s/%s/%s.dat};\n', scenariocolours{j}, ROOT_FOLDER, scenariocodes{j}, shockcode, variablecodes{i});
        end
       fprintf(fid, '\\end{axis}\n');
        if irow<subplotsperrow
            fprintf(fid, '\\&\n');
            irow = irow+1;
        else
            fprintf(fid, '\\\\\n');
            irow = 1;
        end
        if irow~=1 && i==p
            fprintf(fid, '\\\\\n');
        end
    end
    fprintf(fid, '};\n');
    fprintf(fid, '\\end{tikzpicture}\n');
    fprintf(fid, '\\end{document}\n');
    fclose(fid);
else
    warning('Cannot open file %s.tex', filebasename);
end

if compile2pdf
    system(sprintf('pdflatex %s', filebasename))
end