function irf_report = irf_trans_report(baseline_db, irf_db, varargin)
% This function generates the transformed impulse responses to be ploted.
%
% INPUTS
% - baseline_db     [dseries]   Baseline value of the variables.
% - irf_db          [dseries]   Impulse responses of the variables.
% - varargin        [cell]      Accept dev==difference,pdev==% deviations,
%                               qdev==diff log, adev==4th sqrt of diff log
%                               and start_date [dates] denotes the starting
%                               date of the simulation and truncates the
%                               irf_report.
%
%  OUTPUT
%  - irf_report     [dseries]   dseries containing results
%
%  SPECIAL REQUIREMENTS
%  - baseline_db and irf_db are obtained by running irf_single routine

    p = inputParser;

    defaultStartDate = irf_db.dates(1);

    addRequired(p,'baseline_db',@isdseries)
    addRequired(p,'irf_db',@isdseries)
    addParameter(p,'dev',@iscell)
    addParameter(p,'pdev',@iscell)
    addParameter(p,'qdev',@iscell)
    addParameter(p,'adev',@iscell)
    addParameter(p,'ydev',@iscell)
    addParameter(p,'start_date',defaultStartDate,@isdates)


    p.KeepUnmatched = false;
    parse(p,baseline_db,irf_db,varargin{:})

    irf_report_tmp = dseries();

    % Deviations/difference
    for i = 1:length(p.Results.dev)
        irf_report_tmp.(p.Results.dev{i}) = irf_db.(p.Results.dev{i})-baseline_db.(p.Results.dev{i});
    end

    % Percentage deviations
    for i = 1:length(p.Results.pdev)
        irf_report_tmp.(p.Results.pdev{i}) = (irf_db.(p.Results.pdev{i})-baseline_db.(p.Results.pdev{i}))/baseline_db.(p.Results.pdev{i})*100;
    end

    % Quarterly deviations / diff log
    for i = 1:length(p.Results.qdev)
        irf_report_tmp.(p.Results.qdev{i}) = diff(log(irf_db.(p.Results.qdev{i})))*100 - diff(log(baseline_db.(p.Results.qdev{i})))*100;
    end

    % Annualised deviations
    for i = 1:length(p.Results.adev)
        irf_report_tmp.(p.Results.adev{i}) = ((1+diff(log(irf_db.(p.Results.adev{i}))))^4-1)*100 - ((1+diff(log(baseline_db.(p.Results.adev{i}))))^4-1)*100;
    end

    % Annual deviations
    for i = 1:length(p.Results.ydev)
        irf_report_tmp.(p.Results.ydev{i}) = (ydiff(log(irf_db.(p.Results.ydev{i})))-ydiff(log(baseline_db.(p.Results.ydev{i}))))*100;
    end

    irf_report = irf_report_tmp(p.Results.start_date:irf_db.dates(end));

end