function make(varargin)

irfperiods = 120;

ROOT_FOLDER = fileparts(mfilename('fullpath'));
ROOT_FOLDER = ROOT_FOLDER(1:end-7);

if isempty(varargin)
    targets = {'irfs-all'};
else
    if nargin==1 && isequal(varargin{1}, 'all')
        varargin = {'irfs-all', 'forwardguidance-all', 'expected-tfp-growth-shock-all'};
    end
    targets = varargin;
end

% dev (see irf_trans_report) [user defined]
VARIABLES_TO_BE_PLOTTED_AS_DEVIATIONS = {'U2_STN', 'U2_TP_10Y', 'U2_A_YED', 'U2_A_COD', 'U2_G_YER', 'U2_SLRN', 'U2_OIS_10Y', 'U2_HH_SL_CON', 'U2_HH_SL_IHN', 'U2_SCB', 'U2_SCOE', 'U2_HH_L_CON', 'U2_HH_L_IHN', 'U2_LRN', 'U2_HH_L_DEP', 'U2_CB', 'U2_COE', 'U2_R_LUN', 'U2_G_VYER', 'U2_G_HH_Y_VDIR', 'U2_G_HH_S_VDIRL', 'U2_G_HH_S_VDIRT'};

% pdev (see irf_trans_report) [user defined]
VARIABLES_TO_BE_PLOTTED_AS_PERCENTAGE_DEVIATIONS = {'U2_YER', 'U2_HH_COR', 'U2_GO_COR', 'U2_BU_ITR', 'U2_HH_IHR', 'U2_GO_ITR', 'U2_XTR', 'U2_MTR', 'U2_LNN', 'U2_EENX', 'U2_REENX', 'U2_MXR', 'U2_XXR', 'U2_HH_EDIR', 'U2_HH_EDIRL', 'U2_HH_EDIRT', 'U2_HH_DIR', 'U2_HH_DIRL', 'U2_HH_DIRT'};

% qdev (see irf_trans_report) [user defined]
VARIABLES_TO_BE_PLOTTED_AS_LOG_DIFF_DEVIATIONS = {};

% adev (see irf_trans_report) [user defined]
VARIABLES_TO_BE_PLOTTED_AS_ANNUALIZED_LOG_DIFF_DEVIATIONS = {};

% ydev (see irf_trans_report) [user defined]
VARIABLES_TO_BE_PLOTTED_AS_ANNUAL_LOG_DIFF_DEVIATIONS = {};

% Set the list of endogenous variables that will be plotted.
LIST_OF_IRF_VARIABLES = [VARIABLES_TO_BE_PLOTTED_AS_DEVIATIONS, ...
                    VARIABLES_TO_BE_PLOTTED_AS_PERCENTAGE_DEVIATIONS, ...
                    VARIABLES_TO_BE_PLOTTED_AS_LOG_DIFF_DEVIATIONS, ...
                    VARIABLES_TO_BE_PLOTTED_AS_ANNUALIZED_LOG_DIFF_DEVIATIONS, ...
                    VARIABLES_TO_BE_PLOTTED_AS_ANNUAL_LOG_DIFF_DEVIATIONS];

if ismember('irfs-all', targets) || ismember('irfs', targets)

    % Load baseline (used to set the size of the shocks)
    db = dseries('../database/baseline.mat');

    %
    % Compute IRFs
    %

    % FIXME: Would be better if FirstPeriod was defined by M_.orig_maximum_endo_lag+M_.orig_maximum_exo_lag (+1).
    FirstPeriod = 7; % Period of the shock (relative to the baseline):

    % Set list of shocks [user defined]
    LIST_OF_IRF_SHOCKS = {'res_U2_STN', 1.0;
                        'res_U2_TP_10Y', 1.0; ...
                        'res_U2_A_YED', 0.1*db.U2_A_YED.data(FirstPeriod); ...
                        'res_U2_G_C_CER', 0.01; ...
                        'res_U2_Q_T_PRO', 0.01; ...
                        'res_U2_SLRN', 1; ...
                        'res_U2_WDREX', 0.1; ...
                        'res_U2_GO_COR', 0.01*db.U2_YER.data(FirstPeriod); ...
                        'res_U2_CXEDEX', 0.01; ...
                        'res_U2_PCU', 0.01; ...
                        'res_U2_POU', 0.01; ...
                        'res_U2_US_EHIC', 0.01*db.U2_US_EHIC.data(FirstPeriod); ...
                        'res_U2_US_10Y', 0.01*db.U2_US_10Y.data(FirstPeriod); ...
                        'res_U2_FC', 1};

    % Compute backward model IRFs
    for l = 1:rows(LIST_OF_IRF_SHOCKS)
        filename1 = sprintf('%s/models/mat/irfs_backward_model_%s', ROOT_FOLDER, LIST_OF_IRF_SHOCKS{l,1});
        filename2 = sprintf('%s/models/mat/irfs_backward_model_endo_foreign_%s', ROOT_FOLDER, LIST_OF_IRF_SHOCKS{l,1});
        filename3 = sprintf('%s/models/mat/irfs_backward_model_with_exogenous_exchange_rate_%s', ROOT_FOLDER, LIST_OF_IRF_SHOCKS{l,1});
        filename4 = sprintf('%s/models/mat/irfs_backward_model_no_output_gap_effects_on_spreads_%s', ROOT_FOLDER, LIST_OF_IRF_SHOCKS{l,1});
        filename5 = sprintf('%s/models/mat/irfs_backward_model_no_output_gap_effects_on_spreads_and_exogenous_exchange_rate_%s', ROOT_FOLDER, LIST_OF_IRF_SHOCKS{l,1});
        filewext1 = sprintf('%s.mat', filename1);
        filewext2 = sprintf('%s.mat', filename2);
        filewext3 = sprintf('%s.mat', filename3);
        filewext4 = sprintf('%s.mat', filename4);
        filewext5 = sprintf('%s.mat', filename5);
        if ~exist(filewext1, 'file') || ismorerecent(sprintf('%s/models/hybrid.mod', ROOT_FOLDER), filewext1) ...
                || ~exist(filewext2, 'file') || ismorerecent(sprintf('%s/models/hybrid.mod', ROOT_FOLDER), filewext2) ...
                || ~exist(filewext3, 'file') || ismorerecent(sprintf('%s/models/hybrid.mod', ROOT_FOLDER), filewext3) ...
                || ~exist(filewext4, 'file') || ismorerecent(sprintf('%s/models/hybrid.mod', ROOT_FOLDER), filewext4) ...
                || ~exist(filewext5, 'file') || ismorerecent(sprintf('%s/models/hybrid.mod', ROOT_FOLDER), filewext5) ...
                || ismorerecent(sprintf('%s/models/computeirf.m', ROOT_FOLDER), filewext1) ...
                || ismorerecent(sprintf('%s/database/baseline.mat', ROOT_FOLDER), filewext1) ...
                || ismorerecent(sprintf('%s/models/computeirf.m', ROOT_FOLDER), filewext2) ...
                || ismorerecent(sprintf('%s/database/baseline.mat', ROOT_FOLDER), filewext2) ...
                || ismorerecent(sprintf('%s/models/computeirf.m', ROOT_FOLDER), filewext3) ...
                || ismorerecent(sprintf('%s/database/baseline.mat', ROOT_FOLDER), filewext3) ...
                || ismorerecent(sprintf('%s/models/computeirf.m', ROOT_FOLDER), filewext4) ...
                || ismorerecent(sprintf('%s/database/baseline.mat', ROOT_FOLDER), filewext4) ...
                || ismorerecent(sprintf('%s/models/computeirf.m', ROOT_FOLDER), filewext5) ...
                || ismorerecent(sprintf('%s/database/baseline.mat', ROOT_FOLDER), filewext5)
                [ts, ds, us, vs, ws] = computeirf(LIST_OF_IRF_SHOCKS{l,1}, LIST_OF_IRF_SHOCKS{l,2}, LIST_OF_IRF_VARIABLES, 1499);
            ts.save(filename1, 'mat');
            ds.save(filename2, 'mat');
            us.save(filename3, 'mat');
            vs.save(filename4, 'mat');
            ws.save(filename5, 'mat');
            % Normalize backward IRFs (exogenous foreign)
            irfs1 = irf_trans_report(db, ts, 'dev', VARIABLES_TO_BE_PLOTTED_AS_DEVIATIONS, ...
                                     'pdev', VARIABLES_TO_BE_PLOTTED_AS_PERCENTAGE_DEVIATIONS, ...
                                     'qdev', VARIABLES_TO_BE_PLOTTED_AS_LOG_DIFF_DEVIATIONS, ...
                                     'adev', VARIABLES_TO_BE_PLOTTED_AS_ANNUALIZED_LOG_DIFF_DEVIATIONS, ...
                                     'ydev', VARIABLES_TO_BE_PLOTTED_AS_ANNUAL_LOG_DIFF_DEVIATIONS, ...
                                     'start_date', ts.dates(1));
            % Normalize backward IRFs (endogenous foreign)
            irfs2 = irf_trans_report(db, ds, 'dev', VARIABLES_TO_BE_PLOTTED_AS_DEVIATIONS, ...
                                     'pdev', VARIABLES_TO_BE_PLOTTED_AS_PERCENTAGE_DEVIATIONS, ...
                                     'qdev', VARIABLES_TO_BE_PLOTTED_AS_LOG_DIFF_DEVIATIONS, ...
                                     'adev', VARIABLES_TO_BE_PLOTTED_AS_ANNUALIZED_LOG_DIFF_DEVIATIONS, ...
                                     'ydev', VARIABLES_TO_BE_PLOTTED_AS_ANNUAL_LOG_DIFF_DEVIATIONS, ...
                                     'start_date', ds.dates(1));
            % Normalize backward IRFs (exogenous exchange rate)
            irfs3 = irf_trans_report(db, us, 'dev', VARIABLES_TO_BE_PLOTTED_AS_DEVIATIONS, ...
                                     'pdev', VARIABLES_TO_BE_PLOTTED_AS_PERCENTAGE_DEVIATIONS, ...
                                     'qdev', VARIABLES_TO_BE_PLOTTED_AS_LOG_DIFF_DEVIATIONS, ...
                                     'adev', VARIABLES_TO_BE_PLOTTED_AS_ANNUALIZED_LOG_DIFF_DEVIATIONS, ...
                                     'ydev', VARIABLES_TO_BE_PLOTTED_AS_ANNUAL_LOG_DIFF_DEVIATIONS, ...
                                     'start_date', ds.dates(1));
            % Normalize backward IRFs (no output gap effects on spreads)
            irfs4 = irf_trans_report(db, vs, 'dev', VARIABLES_TO_BE_PLOTTED_AS_DEVIATIONS, ...
                                     'pdev', VARIABLES_TO_BE_PLOTTED_AS_PERCENTAGE_DEVIATIONS, ...
                                     'qdev', VARIABLES_TO_BE_PLOTTED_AS_LOG_DIFF_DEVIATIONS, ...
                                     'adev', VARIABLES_TO_BE_PLOTTED_AS_ANNUALIZED_LOG_DIFF_DEVIATIONS, ...
                                     'ydev', VARIABLES_TO_BE_PLOTTED_AS_ANNUAL_LOG_DIFF_DEVIATIONS, ...
                                     'start_date', ds.dates(1));
            irfs5 = irf_trans_report(db, ws, 'dev', VARIABLES_TO_BE_PLOTTED_AS_DEVIATIONS, ...
                                     'pdev', VARIABLES_TO_BE_PLOTTED_AS_PERCENTAGE_DEVIATIONS, ...
                                     'qdev', VARIABLES_TO_BE_PLOTTED_AS_LOG_DIFF_DEVIATIONS, ...
                                     'adev', VARIABLES_TO_BE_PLOTTED_AS_ANNUALIZED_LOG_DIFF_DEVIATIONS, ...
                                     'ydev', VARIABLES_TO_BE_PLOTTED_AS_ANNUAL_LOG_DIFF_DEVIATIONS, ...
                                     'start_date', ds.dates(1));
            if ~exist(sprintf('%s/irfs-data/backward/%s', ROOT_FOLDER, LIST_OF_IRF_SHOCKS{l,1}))
                mkdir(sprintf('%s/irfs-data/backward/%s', ROOT_FOLDER, LIST_OF_IRF_SHOCKS{l,1}));
            end
            if ~exist(sprintf('%s/irfs-data/backward-endo-foreign/%s', ROOT_FOLDER, LIST_OF_IRF_SHOCKS{l,1}))
                mkdir(sprintf('%s/irfs-data/backward-endo-foreign/%s', ROOT_FOLDER, LIST_OF_IRF_SHOCKS{l,1}));
            end
            if ~exist(sprintf('%s/irfs-data/backward-with-exogenous-exchange-rate/%s', ROOT_FOLDER, LIST_OF_IRF_SHOCKS{l,1}))
                mkdir(sprintf('%s/irfs-data/backward-with-exogenous-exchange-rate/%s', ROOT_FOLDER, LIST_OF_IRF_SHOCKS{l,1}));
            end
            if ~exist(sprintf('%s/irfs-data/backward-no-output-gap-effects-on-spreads/%s', ROOT_FOLDER, LIST_OF_IRF_SHOCKS{l,1}))
                mkdir(sprintf('%s/irfs-data/backward-no-output-gap-effects-on-spreads/%s', ROOT_FOLDER, LIST_OF_IRF_SHOCKS{l,1}));
            end
            if ~exist(sprintf('%s/irfs-data/backward-no-output-gap-effects-on-spreads-and-exogenous-exchange-rate/%s', ROOT_FOLDER, LIST_OF_IRF_SHOCKS{l,1}))
                mkdir(sprintf('%s/irfs-data/backward-no-output-gap-effects-on-spreads-and-exogenous-exchange-rate/%s', ROOT_FOLDER, LIST_OF_IRF_SHOCKS{l,1}));
            end
            for i=1:length(LIST_OF_IRF_VARIABLES)
                fid = fopen(sprintf('%s/irfs-data/backward/%s/%s.dat', ROOT_FOLDER, LIST_OF_IRF_SHOCKS{l,1}, LIST_OF_IRF_VARIABLES{i}), 'w');
                fprintf(fid, '%d \t %12.6f\n', [0:irfperiods; transpose(irfs1{LIST_OF_IRF_VARIABLES{i}}.data(1:irfperiods+1))]);
                fclose(fid);
                fid = fopen(sprintf('%s/irfs-data/backward-endo-foreign/%s/%s.dat', ROOT_FOLDER, LIST_OF_IRF_SHOCKS{l,1}, LIST_OF_IRF_VARIABLES{i}), 'w');
                fprintf(fid, '%d \t %12.6f\n', [0:irfperiods; transpose(irfs2{LIST_OF_IRF_VARIABLES{i}}.data(1:irfperiods+1))]);
                fclose(fid);
                fid = fopen(sprintf('%s/irfs-data/backward-with-exogenous-exchange-rate/%s/%s.dat', ROOT_FOLDER, LIST_OF_IRF_SHOCKS{l,1}, LIST_OF_IRF_VARIABLES{i}), 'w');
                fprintf(fid, '%d \t %12.6f\n', [0:irfperiods; transpose(irfs3{LIST_OF_IRF_VARIABLES{i}}.data(1:irfperiods+1))]);
                fclose(fid);
                fid = fopen(sprintf('%s/irfs-data/backward-no-output-gap-effects-on-spreads/%s/%s.dat', ROOT_FOLDER, LIST_OF_IRF_SHOCKS{l,1}, LIST_OF_IRF_VARIABLES{i}), 'w');
                fprintf(fid, '%d \t %12.6f\n', [0:irfperiods; transpose(irfs4{LIST_OF_IRF_VARIABLES{i}}.data(1:irfperiods+1))]);
                fclose(fid);
                fid = fopen(sprintf('%s/irfs-data/backward-no-output-gap-effects-on-spreads-and-exogenous-exchange-rate/%s/%s.dat', ROOT_FOLDER, LIST_OF_IRF_SHOCKS{l,1}, LIST_OF_IRF_VARIABLES{i}), 'w');
                fprintf(fid, '%d \t %12.6f\n', [0:irfperiods; transpose(irfs5{LIST_OF_IRF_VARIABLES{i}}.data(1:irfperiods+1))]);
                fclose(fid);
            end
        end
    end

    % Set macro-variables to be passed to dynare preprocessor.
    IRF_SETUP = {};
    IRF_SETUP{1} = '-DShockNames=[';
    IRF_SETUP{2} = '-DShockValues=[';
    for k=1:rows(LIST_OF_IRF_SHOCKS)-1
        IRF_SETUP{1} = sprintf('%s"%s", ', IRF_SETUP{1}, LIST_OF_IRF_SHOCKS{k,1});
        IRF_SETUP{2} = sprintf('%s%12.6f, ', IRF_SETUP{2}, LIST_OF_IRF_SHOCKS{k,2});
    end
    IRF_SETUP{1} = sprintf('%s"%s"]', IRF_SETUP{1}, LIST_OF_IRF_SHOCKS{end, 1});
    IRF_SETUP{2} = sprintf('%s%12.6f]', IRF_SETUP{2}, LIST_OF_IRF_SHOCKS{end,2});
    IRF_SETUP{3} = '-DVariableNames=[';
    for k=1:length(LIST_OF_IRF_VARIABLES)-1
        IRF_SETUP{3} = sprintf('%s"%s", ', IRF_SETUP{3}, LIST_OF_IRF_VARIABLES{k});
    end
    IRF_SETUP{3} = sprintf('%s"%s"]', IRF_SETUP{3}, LIST_OF_IRF_VARIABLES{end});
    IRF_SETUP{4} = sprintf('-DIrfPeriods=%i',irfperiods);
    IRF_SETUP_LENGTH = length(IRF_SETUP);
    if ~isempty(VARIABLES_TO_BE_PLOTTED_AS_DEVIATIONS)
        IRF_SETUP{IRF_SETUP_LENGTH+1} = '-DVariablePdev=[';
        for k=1:length(VARIABLES_TO_BE_PLOTTED_AS_DEVIATIONS)-1
            IRF_SETUP{IRF_SETUP_LENGTH+1} = sprintf('%s"%s", ', IRF_SETUP{IRF_SETUP_LENGTH+1}, VARIABLES_TO_BE_PLOTTED_AS_DEVIATIONS{k});
        end
        IRF_SETUP{IRF_SETUP_LENGTH+1} = sprintf('%s"%s"]', IRF_SETUP{IRF_SETUP_LENGTH+1}, VARIABLES_TO_BE_PLOTTED_AS_DEVIATIONS{end});
        IRF_SETUP_LENGTH = IRF_SETUP_LENGTH+1;
    end
    if ~isempty(VARIABLES_TO_BE_PLOTTED_AS_PERCENTAGE_DEVIATIONS)
        IRF_SETUP{IRF_SETUP_LENGTH+1} = '-DVariablePpdev=[';
        for k=1:length(VARIABLES_TO_BE_PLOTTED_AS_PERCENTAGE_DEVIATIONS)-1
            IRF_SETUP{IRF_SETUP_LENGTH+1} = sprintf('%s"%s", ', IRF_SETUP{IRF_SETUP_LENGTH+1}, VARIABLES_TO_BE_PLOTTED_AS_PERCENTAGE_DEVIATIONS{k});
        end
        IRF_SETUP{IRF_SETUP_LENGTH+1} = sprintf('%s"%s"]', IRF_SETUP{IRF_SETUP_LENGTH+1}, VARIABLES_TO_BE_PLOTTED_AS_PERCENTAGE_DEVIATIONS{end});
        IRF_SETUP_LENGTH = IRF_SETUP_LENGTH+1;
    end
    if ~isempty(VARIABLES_TO_BE_PLOTTED_AS_LOG_DIFF_DEVIATIONS)
        IRF_SETUP{IRF_SETUP_LENGTH+1} = '-DVariablePqdev=[';
        for k=1:length(VARIABLES_TO_BE_PLOTTED_AS_LOG_DIFF_DEVIATIONS)-1
            IRF_SETUP{IRF_SETUP_LENGTH+1} = sprintf('%s"%s", ', IRF_SETUP{IRF_SETUP_LENGTH+1}, VARIABLES_TO_BE_PLOTTED_AS_LOG_DIFF_DEVIATIONS{k});
        end
        IRF_SETUP{IRF_SETUP_LENGTH+1} = sprintf('%s"%s"]', IRF_SETUP{IRF_SETUP_LENGTH+1}, VARIABLES_TO_BE_PLOTTED_AS_LOG_DIFF_DEVIATIONS{end});
        IRF_SETUP_LENGTH = IRF_SETUP_LENGTH+1;
    end
    if ~isempty(VARIABLES_TO_BE_PLOTTED_AS_ANNUALIZED_LOG_DIFF_DEVIATIONS)
        IRF_SETUP{IRF_SETUP_LENGTH+1} = '-DVariablePadev=[';
        for k=1:length(VARIABLES_TO_BE_PLOTTED_AS_ANNUALIZED_LOG_DIFF_DEVIATIONS)-1
            IRF_SETUP{IRF_SETUP_LENGTH+1} = sprintf('%s"%s", ', IRF_SETUP{IRF_SETUP_LENGTH+1}, VARIABLES_TO_BE_PLOTTED_AS_ANNUALIZED_LOG_DIFF_DEVIATIONS{k});
        end
        IRF_SETUP{IRF_SETUP_LENGTH+1} = sprintf('%s"%s"]', IRF_SETUP{IRF_SETUP_LENGTH+1}, VARIABLES_TO_BE_PLOTTED_AS_ANNUALIZED_LOG_DIFF_DEVIATIONS{end});
        IRF_SETUP_LENGTH = IRF_SETUP_LENGTH+1;
    end
    if ~isempty(VARIABLES_TO_BE_PLOTTED_AS_ANNUALIZED_LOG_DIFF_DEVIATIONS)
        IRF_SETUP{IRF_SETUP_LENGTH+1} = '-DVariablePydev=[';
        for k=1:length(VARIABLES_TO_BE_PLOTTED_AS_ANNUAL_LOG_DIFF_DEVIATIONS)-1
            IRF_SETUP{IRF_SETUP_LENGTH+1} = sprintf('%s"%s", ', IRF_SETUP{IRF_SETUP_LENGTH+1}, VARIABLES_TO_BE_PLOTTED_AS_ANNUAL_LOG_DIFF_DEVIATIONS{k});
        end
        IRF_SETUP{IRF_SETUP_LENGTH+1} = sprintf('%s"%s"]', IRF_SETUP{IRF_SETUP_LENGTH+1}, VARIABLES_TO_BE_PLOTTED_AS_ANNUAL_LOG_DIFF_DEVIATIONS{end});
        IRF_SETUP_LENGTH = IRF_SETUP_LENGTH+1;
    end
    IRF_SETUP{IRF_SETUP_LENGTH+1} = '-DBUILD_IRFS=true';
    IRF_SETUP{IRF_SETUP_LENGTH+2} = '-DBUILD_MATLAB_PLOTS=false';
    IRF_SETUP{IRF_SETUP_LENGTH+2} = '-DOUTPUT_GAP_AS_DEVIATION_TO_BASELINE_IN_SPREADS=true';

    folder = @(x) sprintf('-DSUBFOLDERNAME="%s"', x);

    scenario = 'full';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DFULL=true', folder(scenario), IRF_SETUP{:})
    end

    scenario = 'full-without-output-gap-effects-on-spreads';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DFULL=true', '-DNO_OUTPUT_GAP_EFFECTS_ON_SPREADS=true', folder(scenario), IRF_SETUP{:})
    end

    scenario = 'full-without-output-gap-effects-on-spreads-with-exogenous-exchange-rate';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DDEFAULT_TO_MCE=true', '-DMCE_IN_EXCHANGE_RATE=false', '-DEXOGENOUS_EXCHANGE_RATE=true', '-DNO_OUTPUT_GAP_EFFECTS_ON_SPREADS=true', folder(scenario), IRF_SETUP{:})
    end

    scenario = 'full-with-exogenous-exchange-rate';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DDEFAULT_TO_MCE=true', '-DMCE_IN_EXCHANGE_RATE=false', '-DEXOGENOUS_EXCHANGE_RATE=true', folder(scenario), IRF_SETUP{:})
    end

    scenario = 'full-endo-foreign';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DFULL=true', '-DEXOGENIZE_FOREIGN=false', folder(scenario), IRF_SETUP{:})
    end

    scenario = 'full-without-tp-in-exchange-rate';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DFULL=true', '-DEXCHANGE_RATE_WITH_TP=false', folder(scenario), IRF_SETUP{:})
    end

    scenario = 'full-without-tp-in-exchange-rate-endo-foreign';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DFULL=true', '-DEXCHANGE_RATE_WITH_TP=false', '-DEXOGENIZE_FOREIGN=false', folder(scenario), IRF_SETUP{:})
    end

    scenario = 'without-mce-in-pac';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DDEFAULT_TO_MCE=false', '-DMCE_IN_WAPRO=true', '-DMCE_IN_WAGE=true', '-DMCE_IN_PERMANENT_INCOMES=true', '-DMCE_IN_FINANCIAL=true', '-DMCE_IN_EXCHANGE_RATE=true', folder(scenario), IRF_SETUP{:})
    end

    scenario = 'without-mce-in-pac-with-exogenous-exchange-rate';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DDEFAULT_TO_MCE=false', '-DMCE_IN_WAPRO=true', '-DMCE_IN_WAGE=true', '-DMCE_IN_PERMANENT_INCOMES=true', '-DMCE_IN_FINANCIAL=true', '-DMCE_IN_EXCHANGE_RATE=false', '-DEXOGENOUS_EXCHANGE_RATE=true', folder(scenario), IRF_SETUP{:})
    end

    scenario = 'without-mce-in-pac-endo-foreign';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DDEFAULT_TO_MCE=false', '-DMCE_IN_WAPRO=true', '-DMCE_IN_WAGE=true', '-DMCE_IN_PERMANENT_INCOMES=true', '-DMCE_IN_FINANCIAL=true', '-DMCE_IN_EXCHANGE_RATE=true', '-DEXOGENIZE_FOREIGN=false', folder(scenario), IRF_SETUP{:})
    end

    scenario = 'without-mce-in-exchange-rate';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DDEFAULT_TO_MCE=true ', '-DMCE_IN_EXCHANGE_RATE=false', folder(scenario), IRF_SETUP{:})
    end

    scenario = 'without-mce-in-exchange-rate-endo-foreign';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DDEFAULT_TO_MCE=true ', '-DMCE_IN_EXCHANGE_RATE=false', '-DEXOGENIZE_FOREIGN=false', folder(scenario), IRF_SETUP{:})
    end

    scenario = 'without-mce-in-consumption';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DDEFAULT_TO_MCE=true', '-DMCE_IN_CONSUMPTION=false', folder(scenario), IRF_SETUP{:})
    end

    scenario = 'without-mce-in-consumption-with-exogenous-exchange-rate';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DDEFAULT_TO_MCE=true', '-DMCE_IN_CONSUMPTION=false', '-DMCE_IN_EXCHANGE_RATE=false', '-DEXOGENOUS_EXCHANGE_RATE=true', folder(scenario), IRF_SETUP{:})
    end

    scenario = 'without-mce-in-consumption-endo-foreign';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DDEFAULT_TO_MCE=true', '-DMCE_IN_CONSUMPTION=false', '-DEXOGENIZE_FOREIGN=false', folder(scenario), IRF_SETUP{:})
    end

    scenario = 'without-mce-in-permanent-incomes';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DDEFAULT_TO_MCE=true', '-DMCE_IN_PERMANENT_INCOMES=false', folder(scenario), IRF_SETUP{:})
    end

    scenario = 'without-mce-in-permanent-incomes-with-exogenous-exchange-rate';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DDEFAULT_TO_MCE=true', '-DMCE_IN_PERMANENT_INCOMES=false', '-DMCE_IN_EXCHANGE_RATE=false', '-DEXOGENOUS_EXCHANGE_RATE=true', folder(scenario), IRF_SETUP{:})
    end

    scenario = 'without-mce-in-permanent-incomes-endo-foreign';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DDEFAULT_TO_MCE=true', '-DMCE_IN_PERMANENT_INCOMES=false', '-DEXOGENIZE_FOREIGN=false', folder(scenario), IRF_SETUP{:})
    end

    scenario = 'without-mce-in-pac-consumption';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DDEFAULT_TO_MCE=true', '-DMCE_IN_PAC_CONSUMPTION=false', folder(scenario), IRF_SETUP{:})
    end

    scenario = 'without-mce-in-pac-consumption-with-exogenous-exchange-rate';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DDEFAULT_TO_MCE=true', '-DMCE_IN_PAC_CONSUMPTION=false', '-DMCE_IN_EXCHANGE_RATE=false', '-DEXOGENOUS_EXCHANGE_RATE=true', folder(scenario), IRF_SETUP{:})
    end

    scenario = 'without-mce-in-pac-consumption-endo-foreign';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DDEFAULT_TO_MCE=true', '-DMCE_IN_PAC_CONSUMPTION=false', '-DEXOGENIZE_FOREIGN=false', folder(scenario), IRF_SETUP{:})
    end

    scenario = 'without-mce-in-financial';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DDEFAULT_TO_MCE=true', '-DMCE_IN_FINANCIAL=false', folder(scenario), IRF_SETUP{:})
    end

    scenario = 'without-mce-in-financial-with-exogenous-exchange-rate';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DDEFAULT_TO_MCE=true', '-DMCE_IN_FINANCIAL=false', '-DMCE_IN_EXCHANGE_RATE=false', '-DEXOGENOUS_EXCHANGE_RATE=true', folder(scenario), IRF_SETUP{:})
    end

    scenario = 'without-mce-in-financial-endo-foreign';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DDEFAULT_TO_MCE=true', '-DMCE_IN_FINANCIAL=false', '-DEXOGENIZE_FOREIGN=false', folder(scenario), IRF_SETUP{:})
    end

    scenario = 'without-mce-in-long-term-interest-rate';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DDEFAULT_TO_MCE=true', '-DMCE_IN_LONG_TERM_INTEREST_RATE=false', folder(scenario), IRF_SETUP{:})
    end

    scenario = 'without-mce-in-long-term-interest-rate-with-exogenous-exchange-rate';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DDEFAULT_TO_MCE=true', '-DMCE_IN_LONG_TERM_INTEREST_RATE=false', '-DMCE_IN_EXCHANGE_RATE=false', '-DEXOGENOUS_EXCHANGE_RATE=true', folder(scenario), IRF_SETUP{:})
    end

    scenario = 'without-mce-in-long-term-interest-rate-endo-foreign';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DDEFAULT_TO_MCE=true', '-DMCE_IN_LONG_TERM_INTEREST_RATE=false', '-DEXOGENIZE_FOREIGN=false', folder(scenario), IRF_SETUP{:})
    end

    scenario = 'without-mce-in-expected-averaged-output-gap';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DDEFAULT_TO_MCE=true', '-DMCE_IN_EXPECTED_AVERAGED_OUTPUT_GAP=false', folder(scenario), IRF_SETUP{:})
    end

    scenario = 'without-mce-in-expected-averaged-output-gap-with-exogenous-exchange-rate';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DDEFAULT_TO_MCE=true', '-DMCE_IN_EXPECTED_AVERAGED_OUTPUT_GAP=false', '-DMCE_IN_EXCHANGE_RATE=false', '-DEXOGENOUS_EXCHANGE_RATE=true', folder(scenario), IRF_SETUP{:})
    end

    scenario = 'without-mce-in-expected-averaged-output-gap-endo-foreign';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DDEFAULT_TO_MCE=true', '-DMCE_IN_EXPECTED_AVERAGED_OUTPUT_GAP=false', '-DEXOGENIZE_FOREIGN=false', folder(scenario), IRF_SETUP{:})
    end

    scenario = 'without-mce-in-wapro-and-wage';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DDEFAULT_TO_MCE=true', '-DMCE_IN_WAPRO=false', '-DMCE_IN_WAGE=false', folder(scenario), IRF_SETUP{:})
    end

    scenario = 'without-mce-in-wapro-and-wage-with-exogenous-exchange-rate';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DDEFAULT_TO_MCE=true', '-DMCE_IN_WAPRO=false', '-DMCE_IN_WAGE=false', '-DMCE_IN_EXCHANGE_RATE=false', '-DEXOGENOUS_EXCHANGE_RATE=true', folder(scenario), IRF_SETUP{:})
    end

    scenario = 'without-mce-in-wapro-and-wage-endo-foreign';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DDEFAULT_TO_MCE=true', '-DMCE_IN_WAPRO=false', '-DMCE_IN_WAGE=false', '-DEXOGENIZE_FOREIGN=false', folder(scenario), IRF_SETUP{:})
    end

    scenario = 'without-mce-in-wapro';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DDEFAULT_TO_MCE=true', '-DMCE_IN_WAPRO=false', folder(scenario), IRF_SETUP{:})
    end

    scenario = 'without-mce-in-wapro-with-exogenous-exchange-rate';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DDEFAULT_TO_MCE=true', '-DMCE_IN_WAPRO=false', '-DMCE_IN_EXCHANGE_RATE=false', '-DEXOGENOUS_EXCHANGE_RATE=true', folder(scenario), IRF_SETUP{:})
    end

    scenario = 'without-mce-in-wapro-endo-foreign';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DDEFAULT_TO_MCE=true', '-DMCE_IN_WAPRO=false', '-DEXOGENIZE_FOREIGN=false', folder(scenario), IRF_SETUP{:})
    end

    scenario = 'without-mce-in-wage';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DDEFAULT_TO_MCE=true', '-DMCE_IN_WAGE=false', folder(scenario), IRF_SETUP{:})
    end

    scenario = 'without-mce-in-wage-with-exogenous-exchange-rate';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DDEFAULT_TO_MCE=true', '-DMCE_IN_WAGE=false', '-DMCE_IN_EXCHANGE_RATE=false', '-DEXOGENOUS_EXCHANGE_RATE=true', folder(scenario), IRF_SETUP{:})
    end

    scenario = 'without-mce-in-wage-endo-foreign';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DDEFAULT_TO_MCE=true', '-DMCE_IN_WAGE=false', '-DEXOGENIZE_FOREIGN=false', folder(scenario), IRF_SETUP{:})
    end

    scenario = 'without-mce-in-pac-business';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DDEFAULT_TO_MCE=true', '-DMCE_IN_PAC_BUSINESS=false', folder(scenario), IRF_SETUP{:})
    end

    scenario = 'without-mce-in-pac-business-with-exogenous-exchange-rate';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DDEFAULT_TO_MCE=true', '-DMCE_IN_PAC_BUSINESS=false', '-DMCE_IN_EXCHANGE_RATE=false', '-DEXOGENOUS_EXCHANGE_RATE=true', folder(scenario), IRF_SETUP{:})
    end

    scenario = 'without-mce-in-pac-business-endo-foreign';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DDEFAULT_TO_MCE=true', '-DMCE_IN_PAC_BUSINESS=false', '-DEXOGENIZE_FOREIGN=false', folder(scenario), IRF_SETUP{:})
    end

    scenario = 'without-mce-in-pac-houseprice';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DDEFAULT_TO_MCE=true', '-DMCE_IN_PAC_HOUSEPRICE=false', folder(scenario), IRF_SETUP{:})
    end

    scenario = 'without-mce-in-pac-houseprice-with-exogenous-exchange-rate';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DDEFAULT_TO_MCE=true', '-DMCE_IN_PAC_HOUSEPRICE=false', '-DMCE_IN_EXCHANGE_RATE=false', '-DEXOGENOUS_EXCHANGE_RATE=true', folder(scenario), IRF_SETUP{:})
    end

    scenario = 'without-mce-in-pac-houseprice-endo-foreign';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DDEFAULT_TO_MCE=true', '-DMCE_IN_PAC_HOUSEPRICE=false', '-DEXOGENIZE_FOREIGN=false', folder(scenario), IRF_SETUP{:})
    end

    scenario = 'without-mce-in-pac-property-income';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DDEFAULT_TO_MCE=true', '-DMCE_IN_PAC_PROPERTY_INCOME=false', folder(scenario), IRF_SETUP{:})
    end

    scenario = 'without-mce-in-pac-property-income-with-exogenous-exchange-rate';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DDEFAULT_TO_MCE=true', '-DMCE_IN_PAC_PROPERTY_INCOME=false', '-DMCE_IN_EXCHANGE_RATE=false', '-DEXOGENOUS_EXCHANGE_RATE=true', folder(scenario), IRF_SETUP{:})
    end

    scenario = 'without-mce-in-pac-property-income-endo-foreign';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DDEFAULT_TO_MCE=true', '-DMCE_IN_PAC_PROPERTY_INCOME=false', '-DEXOGENIZE_FOREIGN=false', folder(scenario), IRF_SETUP{:})
    end

    scenario = 'without-mce-in-pac-residential';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DDEFAULT_TO_MCE=true', '-DMCE_IN_PAC_RESIDENTIAL=false', folder(scenario), IRF_SETUP{:})
    end

    scenario = 'without-mce-in-pac-residential-with-exogenous-exchange-rate';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DDEFAULT_TO_MCE=true', '-DMCE_IN_PAC_RESIDENTIAL=false', '-DMCE_IN_EXCHANGE_RATE=false', '-DEXOGENOUS_EXCHANGE_RATE=true', folder(scenario), IRF_SETUP{:})
    end

    scenario = 'without-mce-in-pac-residential-endo-foreign';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DDEFAULT_TO_MCE=true', '-DMCE_IN_PAC_RESIDENTIAL=false', '-DEXOGENIZE_FOREIGN=false', folder(scenario), IRF_SETUP{:})
    end

    scenario = 'without-mce-in-pac-labour';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DDEFAULT_TO_MCE=true', '-DMCE_IN_PAC_LABOUR=false', folder(scenario), IRF_SETUP{:})
    end

    scenario = 'without-mce-in-pac-labour-with-exogenous-exchange-rate';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DDEFAULT_TO_MCE=true', '-DMCE_IN_PAC_LABOUR=false', '-DMCE_IN_EXCHANGE_RATE=false', '-DEXOGENOUS_EXCHANGE_RATE=true', folder(scenario), IRF_SETUP{:})
    end

    scenario = 'without-mce-in-pac-labour-endo-foreign';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DDEFAULT_TO_MCE=true', '-DMCE_IN_PAC_LABOUR=false', '-DEXOGENIZE_FOREIGN=false', folder(scenario), IRF_SETUP{:})
    end

    scenario = 'backward-with-uip';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DBACKWARD_WITH_UIP=true', folder(scenario), IRF_SETUP{:})
    end

    scenario = 'backward-with-uip-endo-foreign';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DBACKWARD_WITH_UIP=true', '-DEXOGENIZE_FOREIGN=false', folder(scenario), IRF_SETUP{:})
    end

    scenario = 'backward-with-uip-without-tp';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DBACKWARD_WITH_UIP=true', '-DEXCHANGE_RATE_WITH_TP=false', folder(scenario), IRF_SETUP{:})
    end

    scenario = 'backward-with-uip-without-tp-endo-foreign';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DBACKWARD_WITH_UIP=true', '-DEXCHANGE_RATE_WITH_TP=false', '-DEXOGENIZE_FOREIGN=false', folder(scenario), IRF_SETUP{:})
    end

    scenario = 'hybrid';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DHYBRID=true', folder(scenario), IRF_SETUP{:})
    end

    scenario = 'hybrid-without-output-gap-effects-on-spreads';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DHYBRID=true', '-DNO_OUTPUT_GAP_EFFECTS_ON_SPREADS=true', folder(scenario), IRF_SETUP{:})
    end

    scenario = 'hybrid-with-exogenous-exchange-rate';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DHYBRID=true', '-DEXOGENOUS_EXCHANGE_RATE=true', folder(scenario), IRF_SETUP{:})
    end

    scenario = 'hybrid-without-output-gap-effects-on-spreads-with-exogenous-exchange-rate';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DHYBRID=true', '-DNO_OUTPUT_GAP_EFFECTS_ON_SPREADS=true', '-DEXOGENOUS_EXCHANGE_RATE=true', folder(scenario), IRF_SETUP{:})
    end

    scenario = 'hybrid-endo-foreign';
    if (ismember('irfs-all', targets) || all(ismember({'irfs',scenario}, targets))) && calldynare(scenario)
        dynare('hybrid', '-DHYBRID=true', '-DEXOGENIZE_FOREIGN=false', folder(scenario), IRF_SETUP{:})
    end

end

if ismember('forwardguidance-all', targets) || ismember('forwardguidance', targets)

    LIST_OF_IRF_SHOCKS = {};

    % Set macro-variables to be passed to dynare preprocessor.
    IRF_SETUP = {};
    IRF_SETUP{1} = '-DVariableNames=[';
    for k=1:length(LIST_OF_IRF_VARIABLES)-1
        IRF_SETUP{1} = sprintf('%s"%s", ', IRF_SETUP{1}, LIST_OF_IRF_VARIABLES{k});
    end
    IRF_SETUP{1} = sprintf('%s"%s"]', IRF_SETUP{1}, LIST_OF_IRF_VARIABLES{end});
    IRF_SETUP{2} = sprintf('-DIrfPeriods=%i',irfperiods);
    IRF_SETUP_LENGTH = length(IRF_SETUP);
    if ~isempty(VARIABLES_TO_BE_PLOTTED_AS_DEVIATIONS)
        IRF_SETUP{IRF_SETUP_LENGTH+1} = '-DVariablePdev=[';
        for k=1:length(VARIABLES_TO_BE_PLOTTED_AS_DEVIATIONS)-1
            IRF_SETUP{IRF_SETUP_LENGTH+1} = sprintf('%s"%s", ', IRF_SETUP{IRF_SETUP_LENGTH+1}, VARIABLES_TO_BE_PLOTTED_AS_DEVIATIONS{k});
        end
        IRF_SETUP{IRF_SETUP_LENGTH+1} = sprintf('%s"%s"]', IRF_SETUP{IRF_SETUP_LENGTH+1}, VARIABLES_TO_BE_PLOTTED_AS_DEVIATIONS{end});
        IRF_SETUP_LENGTH = IRF_SETUP_LENGTH+1;
    end
    if ~isempty(VARIABLES_TO_BE_PLOTTED_AS_PERCENTAGE_DEVIATIONS)
        IRF_SETUP{IRF_SETUP_LENGTH+1} = '-DVariablePpdev=[';
        for k=1:length(VARIABLES_TO_BE_PLOTTED_AS_PERCENTAGE_DEVIATIONS)-1
            IRF_SETUP{IRF_SETUP_LENGTH+1} = sprintf('%s"%s", ', IRF_SETUP{IRF_SETUP_LENGTH+1}, VARIABLES_TO_BE_PLOTTED_AS_PERCENTAGE_DEVIATIONS{k});
        end
        IRF_SETUP{IRF_SETUP_LENGTH+1} = sprintf('%s"%s"]', IRF_SETUP{IRF_SETUP_LENGTH+1}, VARIABLES_TO_BE_PLOTTED_AS_PERCENTAGE_DEVIATIONS{end});
        IRF_SETUP_LENGTH = IRF_SETUP_LENGTH+1;
    end
    if ~isempty(VARIABLES_TO_BE_PLOTTED_AS_LOG_DIFF_DEVIATIONS)
        IRF_SETUP{IRF_SETUP_LENGTH+1} = '-DVariablePqdev=[';
        for k=1:length(VARIABLES_TO_BE_PLOTTED_AS_LOG_DIFF_DEVIATIONS)-1
            IRF_SETUP{IRF_SETUP_LENGTH+1} = sprintf('%s"%s", ', IRF_SETUP{IRF_SETUP_LENGTH+1}, VARIABLES_TO_BE_PLOTTED_AS_LOG_DIFF_DEVIATIONS{k});
        end
        IRF_SETUP{IRF_SETUP_LENGTH+1} = sprintf('%s"%s"]', IRF_SETUP{IRF_SETUP_LENGTH+1}, VARIABLES_TO_BE_PLOTTED_AS_LOG_DIFF_DEVIATIONS{end});
        IRF_SETUP_LENGTH = IRF_SETUP_LENGTH+1;
    end
    if ~isempty(VARIABLES_TO_BE_PLOTTED_AS_ANNUALIZED_LOG_DIFF_DEVIATIONS)
        IRF_SETUP{IRF_SETUP_LENGTH+1} = '-DVariablePadev=[';
        for k=1:length(VARIABLES_TO_BE_PLOTTED_AS_ANNUALIZED_LOG_DIFF_DEVIATIONS)-1
            IRF_SETUP{IRF_SETUP_LENGTH+1} = sprintf('%s"%s", ', IRF_SETUP{IRF_SETUP_LENGTH+1}, VARIABLES_TO_BE_PLOTTED_AS_ANNUALIZED_LOG_DIFF_DEVIATIONS{k});
        end
        IRF_SETUP{IRF_SETUP_LENGTH+1} = sprintf('%s"%s"]', IRF_SETUP{IRF_SETUP_LENGTH+1}, VARIABLES_TO_BE_PLOTTED_AS_ANNUALIZED_LOG_DIFF_DEVIATIONS{end});
        IRF_SETUP_LENGTH = IRF_SETUP_LENGTH+1;
    end
    if ~isempty(VARIABLES_TO_BE_PLOTTED_AS_ANNUALIZED_LOG_DIFF_DEVIATIONS)
        IRF_SETUP{IRF_SETUP_LENGTH+1} = '-DVariablePydev=[';
        for k=1:length(VARIABLES_TO_BE_PLOTTED_AS_ANNUAL_LOG_DIFF_DEVIATIONS)-1
            IRF_SETUP{IRF_SETUP_LENGTH+1} = sprintf('%s"%s", ', IRF_SETUP{IRF_SETUP_LENGTH+1}, VARIABLES_TO_BE_PLOTTED_AS_ANNUAL_LOG_DIFF_DEVIATIONS{k});
        end
        IRF_SETUP{IRF_SETUP_LENGTH+1} = sprintf('%s"%s"]', IRF_SETUP{IRF_SETUP_LENGTH+1}, VARIABLES_TO_BE_PLOTTED_AS_ANNUAL_LOG_DIFF_DEVIATIONS{end});
        IRF_SETUP_LENGTH = IRF_SETUP_LENGTH+1;
    end
    IRF_SETUP{IRF_SETUP_LENGTH+1} = '-DFORWARD_GUIDANCE_PUZZLE=true';
    IRF_SETUP{IRF_SETUP_LENGTH+2} = '-DBUILD_MATLAB_PLOTS=false';

    if (ismember('forwardguidance-all', targets) || all(ismember({'forwardguidance', 'full'}, targets))) && calldynare('forward-guidance/full')
        dynare('hybrid', '-DFULL=true', '-DSUBFOLDERNAME="forward-guidance/full"', IRF_SETUP{:})
    end

    if (ismember('forwardguidance-all', targets) || all(ismember({'forwardguidance', 'without-mce-in-pac'}, targets))) && calldynare('forward-guidance/without-mce-in-pac')
        dynare('hybrid', '-DMCE_IN_WAPRO=true ', '-DMCE_IN_WAGE=true ', '-DMCE_IN_PERMANENT_INCOMES=true ', '-DMCE_IN_FINANCIAL=true ', '-DMCE_IN_EXCHANGE_RATE=true ', '-DDEFAULT_TO_MCE=false', '-DSUBFOLDERNAME="forward-guidance/without-mce-in-pac"', IRF_SETUP{:})
    end

    if (ismember('forwardguidance-all', targets) || all(ismember({'forwardguidance', 'without-mce-in-exchange-rate'}, targets))) && calldynare('forward-guidance/without-mce-in-exchange-rate')
        dynare('hybrid', '-DMCE_IN_EXCHANGE_RATE=false', '-DDEFAULT_TO_MCE=true', '-DSUBFOLDERNAME="forward-guidance/without-mce-in-exchange-rate"', IRF_SETUP{:})
    end

    if (ismember('forwardguidance-all', targets) || all(ismember({'forwardguidance', 'without-mce-in-consumption'}, targets))) && calldynare('forward-guidance/without-mce-in-consumption')
        dynare('hybrid', '-DMCE_IN_CONSUMPTION=false', '-DDEFAULT_TO_MCE=true', '-DSUBFOLDERNAME="forward-guidance/without-mce-in-consumption"', IRF_SETUP{:})
    end

    if (ismember('forwardguidance-all', targets) || all(ismember({'forwardguidance', 'without-mce-in-permanent-incomes'}, targets))) && calldynare('forward-guidance/without-mce-in-permanent-incomes')
        dynare('hybrid', '-DMCE_IN_PERMANENT_INCOMES=false', '-DDEFAULT_TO_MCE=true', '-DSUBFOLDERNAME="forward-guidance/without-mce-in-permanent-incomes"', IRF_SETUP{:})
    end

    if (ismember('forwardguidance-all', targets) || all(ismember({'forwardguidance', 'without-mce-in-pac-consumption'}, targets))) && calldynare('forward-guidance/without-mce-in-pac-consumption')
        dynare('hybrid', '-DMCE_IN_PAC_CONSUMPTION=false', '-DDEFAULT_TO_MCE=true', '-DSUBFOLDERNAME="forward-guidance/without-mce-in-pac-consumption"', IRF_SETUP{:})
    end

    if (ismember('forwardguidance-all', targets) || all(ismember({'forwardguidance', 'without-mce-in-financial'}, targets))) && calldynare('forward-guidance/without-mce-in-financial')
        dynare('hybrid', '-DMCE_IN_FINANCIAL=false', '-DDEFAULT_TO_MCE=true', '-DSUBFOLDERNAME="forward-guidance/without-mce-in-financial"', IRF_SETUP{:})
    end

    if (ismember('forwardguidance-all', targets) || all(ismember({'forwardguidance', 'without-mce-in-long-term-interest-rate'}, targets))) && calldynare('forward-guidance/without-mce-in-long-term-interest-rate')
        dynare('hybrid', '-DMCE_IN_LONG_TERM_INTEREST_RATE=false', '-DDEFAULT_TO_MCE=true', '-DSUBFOLDERNAME="forward-guidance/without-mce-in-long-term-interest-rate"', IRF_SETUP{:})
    end

    if (ismember('forwardguidance-all', targets) || all(ismember({'forwardguidance', 'without-mce-in-expected-averaged-output-gap'}, targets))) && calldynare('forward-guidance/without-mce-in-expected-averaged-output-gap')
        dynare('hybrid', '-DMCE_IN_EXPECTED_AVERAGED_OUTPUT_GAP=false', '-DDEFAULT_TO_MCE=true', '-DSUBFOLDERNAME="forward-guidance/without-mce-in-expected-averaged-output-gap"', IRF_SETUP{:})
    end

    if (ismember('forwardguidance-all', targets) || all(ismember({'forwardguidance', 'without-mce-in-wapro-and-wage'}, targets))) && calldynare('forward-guidance/without-mce-in-wapro-and-wage')
        dynare('hybrid', '-DMCE_IN_WAPRO=false', '-DMCE_IN_WAGE=false', '-DDEFAULT_TO_MCE=true', '-DSUBFOLDERNAME="forward-guidance/without-mce-in-wapro-and-wage"', IRF_SETUP{:})
    end

    if (ismember('forwardguidance-all', targets) || all(ismember({'forwardguidance', 'without-mce-in-wapro'}, targets))) && calldynare('forward-guidance/without-mce-in-wapro')
        dynare('hybrid', '-DMCE_IN_WAPRO=false', '-DDEFAULT_TO_MCE=true', '-DSUBFOLDERNAME="forward-guidance/without-mce-in-wapro"', IRF_SETUP{:})
    end

    if (ismember('forwardguidance-all', targets) || all(ismember({'forwardguidance', 'without-mce-in-wage'}, targets))) && calldynare('forward-guidance/without-mce-in-wage')
        dynare('hybrid', '-DMCE_IN_WAGE=false', '-DDEFAULT_TO_MCE=true', '-DSUBFOLDERNAME="forward-guidance/without-mce-in-wage"', IRF_SETUP{:})
    end

    if (ismember('forwardguidance-all', targets) || all(ismember({'forwardguidance', 'without-mce-in-pac-business'}, targets))) && calldynare('forward-guidance/without-mce-in-pac-business')
        dynare('hybrid', '-DMCE_IN_PAC_BUSINESS=false', '-DDEFAULT_TO_MCE=true', '-DSUBFOLDERNAME="forward-guidance/without-mce-in-pac-business"', IRF_SETUP{:})
    end

    if (ismember('forwardguidance-all', targets) || all(ismember({'forwardguidance', 'without-mce-in-pac-houseprice'}, targets))) && calldynare('forward-guidance/without-mce-in-pac-houseprice')
        dynare('hybrid', '-DMCE_IN_PAC_HOUSEPRICE=false', '-DDEFAULT_TO_MCE=true', '-DSUBFOLDERNAME="forward-guidance/without-mce-in-pac-houseprice"', IRF_SETUP{:})
    end

    if (ismember('forwardguidance-all', targets) || all(ismember({'forwardguidance', 'without-mce-in-pac-property-income'}, targets))) && calldynare('forward-guidance/without-mce-in-pac-property-income')
        dynare('hybrid', '-DMCE_IN_PAC_PROPERTY_INCOME=false', '-DDEFAULT_TO_MCE=true', '-DSUBFOLDERNAME="forward-guidance/without-mce-in-pac-property-income"', IRF_SETUP{:})
    end

    if (ismember('forwardguidance-all', targets) || all(ismember({'forwardguidance', 'without-mce-in-pac-residential'}, targets))) && calldynare('forward-guidance/without-mce-in-pac-residential')
        dynare('hybrid', '-DMCE_IN_PAC_RESIDENTIAL=false', '-DDEFAULT_TO_MCE=true', '-DSUBFOLDERNAME="forward-guidance/without-mce-in-pac-residential"', IRF_SETUP{:})
    end

    if (ismember('forwardguidance-all', targets) || all(ismember({'forwardguidance', 'without-mce-in-pac-labour'}, targets))) && calldynare('forward-guidance/without-mce-in-pac-labour')
        dynare('hybrid', '-DMCE_IN_PAC_LABOUR=false', '-DDEFAULT_TO_MCE=true', '-DSUBFOLDERNAME="forward-guidance/without-mce-in-pac-labour"', IRF_SETUP{:})
    end

    if (ismember('forwardguidance-all', targets) || all(ismember({'forwardguidance', 'backward-with-uip'}, targets))) && calldynare('forward-guidance/backward-with-uip')
        dynare('hybrid', '-DBACKWARD_WITH_UIP=true', '-DSUBFOLDERNAME="forward-guidance/backward-with-uip"', IRF_SETUP{:})
    end

    if (ismember('forwardguidance-all', targets) || all(ismember({'forwardguidance', 'hybrid'}, targets))) && calldynare('forward-guidance/hybrid')
        dynare('hybrid', '-DHYBRID=true', '-DSUBFOLDERNAME="forward-guidance/hybrid"', IRF_SETUP{:})
    end

end

if ismember('expected-tfp-growth-shock-all', targets) || ismember('expected-tfp-growth-shock', targets)

    LIST_OF_IRF_SHOCKS = {};

    % Set macro-variables to be passed to dynare preprocessor.
    IRF_SETUP = {};
    IRF_SETUP{1} = '-DVariableNames=[';
    for k=1:length(LIST_OF_IRF_VARIABLES)-1
        IRF_SETUP{1} = sprintf('%s"%s", ', IRF_SETUP{1}, LIST_OF_IRF_VARIABLES{k});
    end
    IRF_SETUP{1} = sprintf('%s"%s"]', IRF_SETUP{1}, LIST_OF_IRF_VARIABLES{end});
    IRF_SETUP{2} = sprintf('-DIrfPeriods=%i',irfperiods);
    IRF_SETUP_LENGTH = length(IRF_SETUP);
    if ~isempty(VARIABLES_TO_BE_PLOTTED_AS_DEVIATIONS)
        IRF_SETUP{IRF_SETUP_LENGTH+1} = '-DVariablePdev=[';
        for k=1:length(VARIABLES_TO_BE_PLOTTED_AS_DEVIATIONS)-1
            IRF_SETUP{IRF_SETUP_LENGTH+1} = sprintf('%s"%s", ', IRF_SETUP{IRF_SETUP_LENGTH+1}, VARIABLES_TO_BE_PLOTTED_AS_DEVIATIONS{k});
        end
        IRF_SETUP{IRF_SETUP_LENGTH+1} = sprintf('%s"%s"]', IRF_SETUP{IRF_SETUP_LENGTH+1}, VARIABLES_TO_BE_PLOTTED_AS_DEVIATIONS{end});
        IRF_SETUP_LENGTH = IRF_SETUP_LENGTH+1;
    end
    if ~isempty(VARIABLES_TO_BE_PLOTTED_AS_PERCENTAGE_DEVIATIONS)
        IRF_SETUP{IRF_SETUP_LENGTH+1} = '-DVariablePpdev=[';
        for k=1:length(VARIABLES_TO_BE_PLOTTED_AS_PERCENTAGE_DEVIATIONS)-1
            IRF_SETUP{IRF_SETUP_LENGTH+1} = sprintf('%s"%s", ', IRF_SETUP{IRF_SETUP_LENGTH+1}, VARIABLES_TO_BE_PLOTTED_AS_PERCENTAGE_DEVIATIONS{k});
        end
        IRF_SETUP{IRF_SETUP_LENGTH+1} = sprintf('%s"%s"]', IRF_SETUP{IRF_SETUP_LENGTH+1}, VARIABLES_TO_BE_PLOTTED_AS_PERCENTAGE_DEVIATIONS{end});
        IRF_SETUP_LENGTH = IRF_SETUP_LENGTH+1;
    end
    if ~isempty(VARIABLES_TO_BE_PLOTTED_AS_LOG_DIFF_DEVIATIONS)
        IRF_SETUP{IRF_SETUP_LENGTH+1} = '-DVariablePqdev=[';
        for k=1:length(VARIABLES_TO_BE_PLOTTED_AS_LOG_DIFF_DEVIATIONS)-1
            IRF_SETUP{IRF_SETUP_LENGTH+1} = sprintf('%s"%s", ', IRF_SETUP{IRF_SETUP_LENGTH+1}, VARIABLES_TO_BE_PLOTTED_AS_LOG_DIFF_DEVIATIONS{k});
        end
        IRF_SETUP{IRF_SETUP_LENGTH+1} = sprintf('%s"%s"]', IRF_SETUP{IRF_SETUP_LENGTH+1}, VARIABLES_TO_BE_PLOTTED_AS_LOG_DIFF_DEVIATIONS{end});
        IRF_SETUP_LENGTH = IRF_SETUP_LENGTH+1;
    end
    if ~isempty(VARIABLES_TO_BE_PLOTTED_AS_ANNUALIZED_LOG_DIFF_DEVIATIONS)
        IRF_SETUP{IRF_SETUP_LENGTH+1} = '-DVariablePadev=[';
        for k=1:length(VARIABLES_TO_BE_PLOTTED_AS_ANNUALIZED_LOG_DIFF_DEVIATIONS)-1
            IRF_SETUP{IRF_SETUP_LENGTH+1} = sprintf('%s"%s", ', IRF_SETUP{IRF_SETUP_LENGTH+1}, VARIABLES_TO_BE_PLOTTED_AS_ANNUALIZED_LOG_DIFF_DEVIATIONS{k});
        end
        IRF_SETUP{IRF_SETUP_LENGTH+1} = sprintf('%s"%s"]', IRF_SETUP{IRF_SETUP_LENGTH+1}, VARIABLES_TO_BE_PLOTTED_AS_ANNUALIZED_LOG_DIFF_DEVIATIONS{end});
        IRF_SETUP_LENGTH = IRF_SETUP_LENGTH+1;
    end
    if ~isempty(VARIABLES_TO_BE_PLOTTED_AS_ANNUALIZED_LOG_DIFF_DEVIATIONS)
        IRF_SETUP{IRF_SETUP_LENGTH+1} = '-DVariablePydev=[';
        for k=1:length(VARIABLES_TO_BE_PLOTTED_AS_ANNUAL_LOG_DIFF_DEVIATIONS)-1
            IRF_SETUP{IRF_SETUP_LENGTH+1} = sprintf('%s"%s", ', IRF_SETUP{IRF_SETUP_LENGTH+1}, VARIABLES_TO_BE_PLOTTED_AS_ANNUAL_LOG_DIFF_DEVIATIONS{k});
        end
        IRF_SETUP{IRF_SETUP_LENGTH+1} = sprintf('%s"%s"]', IRF_SETUP{IRF_SETUP_LENGTH+1}, VARIABLES_TO_BE_PLOTTED_AS_ANNUAL_LOG_DIFF_DEVIATIONS{end});
        IRF_SETUP_LENGTH = IRF_SETUP_LENGTH+1;
    end
    IRF_SETUP{IRF_SETUP_LENGTH+1} = '-DEXPECTED_PRODUCTIVITY_SHOCK=true';
    IRF_SETUP{IRF_SETUP_LENGTH+2} = '-DBUILD_MATLAB_PLOTS=false';

    if (ismember('expected-tfp-growth-shock-all', targets) || all(ismember({'expected-tfp-growth-shock', 'full'}, targets))) && calldynare('tfp-growth-expected-shock/full')
        dynare('hybrid', '-DFULL=true', '-DSUBFOLDERNAME="tfp-growth-expected-shock/full"', IRF_SETUP{:})
    end

    if (ismember('expected-tfp-growth-shock-all', targets) || all(ismember({'expected-tfp-growth-shock', 'without-mce-in-pac'}, targets))) && calldynare('tfp-growth-expected-shock/without-mce-in-pac')
        dynare('hybrid', '-DDEFAULT_TO_MCE=false', '-DMCE_IN_WAPRO=true', '-DMCE_IN_WAGE=true', '-DMCE_IN_PERMANENT_INCOMES=true', '-DMCE_IN_FINANCIAL=true', '-DMCE_IN_EXCHANGE_RATE=true', ...
               '-DSUBFOLDERNAME="tfp-growth-expected-shock/without-mce-in-pac"', IRF_SETUP{:})
    end

    if (ismember('expected-tfp-growth-shock-all', targets) || all(ismember({'expected-tfp-growth-shock', 'without-mce-in-exchange-rate'}, targets))) && calldynare('tfp-growth-expected-shock/without-mce-in-exchange-rate')
        dynare('hybrid', '-DMCE_IN_EXCHANGE_RATE=false', '-DDEFAULT_TO_MCE=true', '-DSUBFOLDERNAME="tfp-growth-expected-shock/without-mce-in-exchange-rate"', IRF_SETUP{:})
    end

    if (ismember('expected-tfp-growth-shock-all', targets) || all(ismember({'expected-tfp-growth-shock', 'without-mce-in-consumption'}, targets))) && calldynare('tfp-growth-expected-shock/without-mce-in-consumption')
        dynare('hybrid', '-DMCE_IN_CONSUMPTION=false', '-DDEFAULT_TO_MCE=true', '-DSUBFOLDERNAME="tfp-growth-expected-shock/without-mce-in-consumption"', IRF_SETUP{:})
    end

    if (ismember('expected-tfp-growth-shock-all', targets) || all(ismember({'expected-tfp-growth-shock', 'without-mce-in-permanent-incomes'}, targets))) && calldynare('tfp-growth-expected-shock/without-mce-in-permanent-incomes')
        dynare('hybrid', '-DMCE_IN_PERMANENT_INCOMES=false', '-DDEFAULT_TO_MCE=true', '-DSUBFOLDERNAME="tfp-growth-expected-shock/without-mce-in-permanent-incomes"', IRF_SETUP{:})
    end

    if (ismember('expected-tfp-growth-shock-all', targets) || all(ismember({'expected-tfp-growth-shock', 'without-mce-in-pac-consumption'}, targets))) && calldynare('tfp-growth-expected-shock/without-mce-in-pac-consumption')
        dynare('hybrid', '-DMCE_IN_PAC_CONSUMPTION=false', '-DDEFAULT_TO_MCE=true', '-DSUBFOLDERNAME="tfp-growth-expected-shock/without-mce-in-pac-consumption"', IRF_SETUP{:})
    end

    if (ismember('expected-tfp-growth-shock-all', targets) || all(ismember({'expected-tfp-growth-shock', 'without-mce-in-financial'}, targets))) && calldynare('tfp-growth-expected-shock/without-mce-in-financial')
        dynare('hybrid', '-DMCE_IN_FINANCIAL=false', '-DDEFAULT_TO_MCE=true', '-DSUBFOLDERNAME="tfp-growth-expected-shock/without-mce-in-financial"', IRF_SETUP{:})
    end

    % This one is not working with the latest calibration provided by Nikola
    %
    % if (ismember('expected-tfp-growth-shock-all', targets) || all(ismember({'expected-tfp-growth-shock', 'without-mce-in-long-term-interest-rate'}, targets))) && calldynare('tfp-growth-expected-shock/without-mce-in-long-term-interest-rate')
    %     dynare('hybrid', '-DMCE_IN_LONG_TERM_INTEREST_RATE=false', '-DDEFAULT_TO_MCE=true', '-DSUBFOLDERNAME="tfp-growth-expected-shock/without-mce-in-long-term-interest-rate"', IRF_SETUP{:})
    % end

    if (ismember('expected-tfp-growth-shock-all', targets) || all(ismember({'expected-tfp-growth-shock', 'without-mce-in-expected-averages-output-gap'}, targets))) && calldynare('tfp-growth-expected-shock/without-mce-in-expected-averaged-output-gap')
        dynare('hybrid', '-DMCE_IN_EXPECTED_AVERAGED_OUTPUT_GAP=false', '-DDEFAULT_TO_MCE=true', '-DSUBFOLDERNAME="tfp-growth-expected-shock/without-mce-in-expected-averaged-output-gap"', IRF_SETUP{:})
    end

    if (ismember('expected-tfp-growth-shock-all', targets) || all(ismember({'expected-tfp-growth-shock', 'without-mce-in-wapro-and-wage'}, targets))) && calldynare('tfp-growth-expected-shock/without-mce-in-wapro-and-wage')
        dynare('hybrid', '-DMCE_IN_WAPRO=false', '-DMCE_IN_WAGE=false', '-DDEFAULT_TO_MCE=true', '-DSUBFOLDERNAME="tfp-growth-expected-shock/without-mce-in-wapro-and-wage"', IRF_SETUP{:})
    end

    if (ismember('expected-tfp-growth-shock-all', targets) || all(ismember({'expected-tfp-growth-shock', 'without-mce-in-wapro'}, targets))) && calldynare('tfp-growth-expected-shock/without-mce-in-wapro')
        dynare('hybrid', '-DMCE_IN_WAPRO=false', '-DDEFAULT_TO_MCE=true', '-DSUBFOLDERNAME="tfp-growth-expected-shock/without-mce-in-wapro"', IRF_SETUP{:})
    end

    if (ismember('expected-tfp-growth-shock-all', targets) || all(ismember({'expected-tfp-growth-shock', 'without-mce-in-wage'}, targets))) && calldynare('tfp-growth-expected-shock/without-mce-in-wage')
        dynare('hybrid', '-DMCE_IN_WAGE=false', '-DDEFAULT_TO_MCE=true', '-DSUBFOLDERNAME="tfp-growth-expected-shock/without-mce-in-wage"', IRF_SETUP{:})
    end

    if (ismember('expected-tfp-growth-shock-all', targets) || all(ismember({'expected-tfp-growth-shock', 'without-mce-in-pac-business'}, targets))) && calldynare('tfp-growth-expected-shock/without-mce-in-pac-business')
        dynare('hybrid', '-DMCE_IN_PAC_BUSINESS=false', '-DDEFAULT_TO_MCE=true', '-DSUBFOLDERNAME="tfp-growth-expected-shock/without-mce-in-pac-business"', IRF_SETUP{:})
    end

    if (ismember('expected-tfp-growth-shock-all', targets) || all(ismember({'expected-tfp-growth-shock', 'without-mce-in-pac-houseprice'}, targets))) && calldynare('tfp-growth-expected-shock/without-mce-in-pac-houseprice')
        dynare('hybrid', '-DMCE_IN_PAC_HOUSEPRICE=false', '-DDEFAULT_TO_MCE=true', '-DSUBFOLDERNAME="tfp-growth-expected-shock/without-mce-in-pac-houseprice"', IRF_SETUP{:})
    end

    if (ismember('expected-tfp-growth-shock-all', targets) || all(ismember({'expected-tfp-growth-shock', 'without-mce-in-property-income'}, targets))) && calldynare('tfp-growth-expected-shock/without-mce-in-pac-property-income')
        dynare('hybrid', '-DMCE_IN_PAC_PROPERTY_INCOME=false', '-DDEFAULT_TO_MCE=true', '-DSUBFOLDERNAME="tfp-growth-expected-shock/without-mce-in-pac-property-income"', IRF_SETUP{:})
    end

    if (ismember('expected-tfp-growth-shock-all', targets) || all(ismember({'expected-tfp-growth-shock', 'without-mce-in-pac-residential'}, targets))) && calldynare('tfp-growth-expected-shock/without-mce-in-pac-residential')
        dynare('hybrid', '-DMCE_IN_PAC_RESIDENTIAL=false', '-DDEFAULT_TO_MCE=true', '-DSUBFOLDERNAME="tfp-growth-expected-shock/without-mce-in-pac-residential"', IRF_SETUP{:})
    end

    if (ismember('expected-tfp-growth-shock-all', targets) || all(ismember({'expected-tfp-growth-shock', 'without-mce-in-pac-labour'}, targets))) && calldynare('tfp-growth-expected-shock/without-mce-in-pac-labour')
        dynare('hybrid', '-DMCE_IN_PAC_LABOUR=false', '-DDEFAULT_TO_MCE=true', '-DSUBFOLDERNAME="tfp-growth-expected-shock/without-mce-in-pac-labour"', IRF_SETUP{:})
    end

    if (ismember('expected-tfp-growth-shock-all', targets) || all(ismember({'expected-tfp-growth-shock', 'backward-with-uip'}, targets))) && calldynare('tfp-growth-expected-shock/backward-with-uip')
        dynare('hybrid', '-DBACKWARD_WITH_UIP=true', '-DDEFAULT_TO_MCE=false', '-DSUBFOLDERNAME="tfp-growth-expected-shock/backward-with-uip"', IRF_SETUP{:})
    end

    if (ismember('expected-tfp-growth-shock-all', targets) || all(ismember({'expected-tfp-growth-shock', 'hybrid'}, targets))) && calldynare('tfp-growth-expected-shock/hybrid')
        dynare('hybrid', '-DHYBRID=true', '-DSUBFOLDERNAME="tfp-growth-expected-shock/hybrid"', IRF_SETUP{:})
    end

end


function b = calldynare(subfolder)
    b = false;
    if isempty(LIST_OF_IRF_SHOCKS)
        for j=1:length(LIST_OF_IRF_VARIABLES)
            b = ~isfile(sprintf('%s/irfs-data/hybrid/%s/%s.dat', ROOT_FOLDER, subfolder, LIST_OF_IRF_VARIABLES{j})) || ...
                ismorerecent(sprintf('%s/models/hybrid.mod', ROOT_FOLDER), sprintf('%s/irfs-data/hybrid/%s/%s.dat', ROOT_FOLDER, subfolder, LIST_OF_IRF_VARIABLES{j})) || ...
                ismorerecent(sprintf('%s/models/core.mod', ROOT_FOLDER), sprintf('%s/irfs-data/hybrid/%s/%s.dat', ROOT_FOLDER, subfolder, LIST_OF_IRF_VARIABLES{j})) || ...
                ismorerecent(sprintf('%s/models/update-database.mod', ROOT_FOLDER), sprintf('%s/irfs-data/hybrid/%s/%s.dat', ROOT_FOLDER, subfolder, LIST_OF_IRF_VARIABLES{j})) || ...
                ismorerecent(sprintf('%s/database/baseline.mat', ROOT_FOLDER), sprintf('%s/irfs-data/hybrid/%s/%s.dat', ROOT_FOLDER, subfolder, LIST_OF_IRF_VARIABLES{j}));
            if b
                return
            end
        end
    else
        for i=1:length(LIST_OF_IRF_SHOCKS)
            for j=1:length(LIST_OF_IRF_VARIABLES)
                b = ~isfile(sprintf('%s/irfs-data/hybrid/%s/%s/%s.dat', ROOT_FOLDER, subfolder, LIST_OF_IRF_SHOCKS{i}, LIST_OF_IRF_VARIABLES{j})) || ...
                    ismorerecent(sprintf('%s/models/hybrid.mod', ROOT_FOLDER), sprintf('%s/irfs-data/hybrid/%s/%s/%s.dat', ROOT_FOLDER, subfolder, LIST_OF_IRF_SHOCKS{i}, LIST_OF_IRF_VARIABLES{j})) || ...
                    ismorerecent(sprintf('%s/models/core.mod', ROOT_FOLDER), sprintf('%s/irfs-data/hybrid/%s/%s/%s.dat', ROOT_FOLDER, subfolder, LIST_OF_IRF_SHOCKS{i}, LIST_OF_IRF_VARIABLES{j})) || ...
                    ismorerecent(sprintf('%s/models/update-database.mod', ROOT_FOLDER), sprintf('%s/irfs-data/hybrid/%s/%s/%s.dat', ROOT_FOLDER, subfolder, LIST_OF_IRF_SHOCKS{i}, LIST_OF_IRF_VARIABLES{j})) || ...
                    ismorerecent(sprintf('%s/database/baseline.mat', ROOT_FOLDER), sprintf('%s/irfs-data/hybrid/%s/%s/%s.dat', ROOT_FOLDER, subfolder, LIST_OF_IRF_SHOCKS{i}, LIST_OF_IRF_VARIABLES{j}));
                if b
                    return
                end
            end
        end
    end
end

end